<?php
/**
 * Created by IntelliJ IDEA.
 * User: damien
 * Date: 11/02/2017
 * Time: 16:10
 */

namespace CloudFileStorage\Adapter;

use CloudFileStorage\CacheUtil;
use CloudFileStorage\AdapterFile;
use S3;

class S3CloudFile implements AdapterFile
{
    private $object,$bucket, $objectStorageService, $metadataFetched=false;
    private $isDebug=false;
    /** @var CacheUtil $cacheUtil */
    private $cacheUtil;

    public function __construct($containerObject, S3Cloud $objectStorageService,$options)
    {
        if($options!= "" && is_array($options) && isset($options["isDebug"]) && $options["isDebug"]==true)
            $this->isDebug=true;
        if($options!= "" && is_array($options) && isset($options["cacheUtil"]))
            $this->cacheUtil=$options["cacheUtil"];
        $this->object = $containerObject;
        $this->objectStorageService = $objectStorageService;
        $this->bucket = $objectStorageService->bucket;
    }

    public function isDir()
    {
        if(isset($this->object["prefix"]))
            return true;
    }

    public function getKey()
    {
        if(isset($this->object["prefix"]))
            return substr_replace($this->object["prefix"] ,"", -1);
        return $this->object["name"];
    }

    public function getSize()
    {
        if(isset($this->object["size"]))
            return $this->object["size"];
        return 0;
    }

    public function getMimeType()
    {
        if($this->isDir())
            return "";
        static $exts = array(
            'jpg' => 'image/jpeg', 'jpeg' => 'image/jpeg', 'gif' => 'image/gif',
            'png' => 'image/png', 'ico' => 'image/x-icon', 'pdf' => 'application/pdf',
            'tif' => 'image/tiff', 'tiff' => 'image/tiff', 'svg' => 'image/svg+xml',
            'svgz' => 'image/svg+xml', 'swf' => 'application/x-shockwave-flash',
            'zip' => 'application/zip', 'gz' => 'application/x-gzip',
            'tar' => 'application/x-tar', 'bz' => 'application/x-bzip',
            'bz2' => 'application/x-bzip2',  'rar' => 'application/x-rar-compressed',
            'exe' => 'application/x-msdownload', 'msi' => 'application/x-msdownload',
            'cab' => 'application/vnd.ms-cab-compressed', 'txt' => 'text/plain',
            'asc' => 'text/plain', 'htm' => 'text/html', 'html' => 'text/html',
            'css' => 'text/css', 'js' => 'text/javascript',
            'xml' => 'text/xml', 'xsl' => 'application/xsl+xml',
            'ogg' => 'application/ogg', 'mp3' => 'audio/mpeg', 'wav' => 'audio/x-wav',
            'avi' => 'video/x-msvideo', 'mpg' => 'video/mpeg', 'mpeg' => 'video/mpeg',
            'mov' => 'video/quicktime', 'flv' => 'video/x-flv', 'php' => 'text/x-php'
        );

        $ext = strtolower(pathinfo($this->getName($this->getKey()), PATHINFO_EXTENSION));
        if (isset($exts[$ext])) return $exts[$ext];

        return "";
    }

    public function getLastModified()
    {
        if(isset($this->object["time"]))
            return $this->object["time"];
        return false;
    }

    public function getName($key)
    {
        $tmp = explode("/",$key);
        if($this->isDir()){
            $nom = $tmp[count($tmp)-1];
        }else{
            $nom = end($tmp);
        }
        return $nom;
    }
    private function startsWith($haystack, $needle)
    {
        $length = strlen($needle);
        return (substr($haystack, 0, $length) === $needle);
    }


    private function updateMetadata($force = array()){

        if($this->isDebug)
            $start=microtime(true);

        if($this->startsWith($this->getKey(),"thumb") || $this->isDir()) //pas besoin de metadata pour thumb et dossiers
            return;


        if($this->cacheUtil != null && $this->cacheUtil->cacheIsValid(CacheUtil::CACHE_GETMETADATAFILE_OPERATION,$this->getKey(),36*60)){
            $metadata =  $this->cacheUtil->getCacheValue(CacheUtil::CACHE_GETMETADATAFILE_OPERATION,$this->getKey());
            $this->object["metadata"] = $metadata;
            // var_dump($metadata);

            if($this->isDebug){
                $end = microtime(true) - $start;
                echo "OpenCloudFile updateMetadata (cached): ". (round($end,4)*1000)."ms\n<br/>";
            }
            return;
        }
        if(!$this->metadataFetched){
            $this->object["metadata"]= [];
            $headers = $this->objectStorageService->client->getObjectInfo($this->bucket,$this->getKey());

            foreach ($headers as $headerName=>$headerValue){
                if($this->startsWith($headerName,"x-amz-meta-")&&$headerName != "x-amz-meta-etag"&&$headerName != "x-amz-meta-request-id"){
                    $this->object["metadata"][substr($headerName,11,strlen($headerName))]=$headerValue;
                }
            }
            $this->metadataFetched = true;
            $objectMetadata = $this->object["metadata"];
        }
        $localFileLocaltion = tempnam(sys_get_temp_dir(),$this->getKey().'/'.$this->getName($this->getKey()));
        $tmp = explode(".",$this->getName($this->getKey()));
        $extension =  end($tmp);
        $downloaded = false;
        $asNewMetaData=false;

        if( (isset($objectMetadata["dimensionheight"]) == false && $this->startsWith($this->getMimeType(),"image")) || (isset($force["taille"]) && $force["taille"]=true) ){
            //il faut calculé la taille de l'image et mettre la méta data. C'est important pour l'arbo dans albulle
            if($downloaded == false){
                $this->download($localFileLocaltion);
                $downloaded = true;
            }

            if($extension == "svg"){
                $asNewMetaData = true;
                $objectMetadata['dimensionwidth']=0;
                $objectMetadata['dimensionheight']=0;
            }else{
                list($image_with,$image_height) = getimagesize($localFileLocaltion);
                $asNewMetaData = true;
                $objectMetadata['dimensionwidth']=$image_with;
                $objectMetadata['dimensionheight']=$image_height;
            }

        }

        // Ici on s'occupe de la miniature

        if(isset($objectMetadata["thumbpath"]) == false && $this->startsWith($this->getMimeType(),"image") || (isset($force["thumb"]) && $force["thumb"]=true)){
            if($downloaded == false){
                $this->download($localFileLocaltion);
                $downloaded = true;
            }
            $localFileLocaltionThumb = tempnam(sys_get_temp_dir(),"thumb/".$this->getName($this->getKey()));
            $tmp = explode(".", $this->getName($this->getKey()));
            $format = array_pop($tmp);

            if(file_exists($localFileLocaltionThumb))
                unlink($localFileLocaltionThumb);

            if($format != "pdf"){
                $this->createThumbnail( $localFileLocaltion, $localFileLocaltionThumb);

                $tmpPathThumb = str_replace("/","_",$this->getKey());
                $pathThumb = "thumb/".$tmpPathThumb;

                $this->objectStorageService->uploadLocalFile($pathThumb, $localFileLocaltionThumb);

                if(file_exists($localFileLocaltionThumb))
                    unlink($localFileLocaltionThumb);
                $asNewMetaData = true;

                $objectMetadata['thumbpath']=$pathThumb;
            }

        }

        // On met à jour les metadonnées
        if(file_exists($localFileLocaltion))
            unlink($localFileLocaltion);


        if($asNewMetaData){
            $this->object["metadata"] = $objectMetadata;
            $this->objectStorageService->client->copyObject($this->bucket,$this->getKey(),$this->bucket,$this->getKey(),S3::ACL_PRIVATE,$objectMetadata,array("Content-Type"=>$this->getMimeType()));
        }
        if(isset($objectMetadata["thumbpath"]) && isset($objectMetadata["dimensionheight"]) && $this->cacheUtil != null){
            $this->cacheUtil->saveCacheValue(CacheUtil::CACHE_GETMETADATAFILE_OPERATION,$this->getKey(),$objectMetadata);
        }
        if($this->isDebug){
            $end = microtime(true) - $start;
            echo "S3CloudFile updateMetadata: ". (round($end,4)*1000)."ms\n<br/>";
        }
    }

    private function download($localFileLocaltion)
    {

        //Telechargement fichier pour recuperer les taille et mettre les metadata des images
        $out = fopen($localFileLocaltion,"wb");
        if ($out == FALSE){
            print "Probléme mise à jour metadata objet<br>";
        }

        $ch = curl_init();

        curl_setopt($ch, CURLOPT_FILE, $out);
        curl_setopt($ch, CURLOPT_HEADER, 0);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        curl_setopt($ch, CURLOPT_URL, $this->getPublicUrl());
        curl_exec($ch);
        curl_close($ch);
    }

    public function getImageWidthHeight(){
        $this->updateMetadata();

        return array(
            $this->object["metadata"]["dimensionwidth"],
            $this->object["metadata"]["dimensionheight"],
        );
    }

    public function getThumbObjet($retry=true){
        $this->updateMetadata();

        try{
            $thumbObj = $this->object["metadata"]["thumbpath"];
            return $thumbObj;
        }catch (Exception $e){
            //fichier existe pas il semblerait
            if($retry){
                $this->updateMetadata(array("thumb"=>true));
                return $this->getThumbObjet(false);
            }
            return null;
        }
    }
    public function getPublicUrl(){
        $dateExpiration = new \DateTime();
        $dateExpiration->setTime(23, 59, 59);
        $tmpUrl = $this->objectStorageService->generateTemporyUrl($this->getKey(), $dateExpiration, $method="GET",$forceDownload=false);

        return $tmpUrl;
    }

    public function createThumbnail($image_src , $image_dest = NULL , $max_size = 150, $expand = FALSE, $square = FALSE){
        if( !file_exists($image_src) ) return FALSE;
        $fileinfo = getimagesize($image_src);
        if( !$fileinfo ) return FALSE;
        $width     = $fileinfo[0];
        $height    = $fileinfo[1];
        $type_mime = $fileinfo['mime'];
        $type      = str_replace('image/', '', $type_mime);
        if( !$expand && max($width, $height)<=$max_size && (!$square || ($square && $width==$height) ) ){
            if($image_dest) {
                return copy($image_src, $image_dest);
            } else {
                header('Content-Type: '. $type_mime);
                return (boolean) readfile($image_src);
            }
        }
        $ratio = $width / $height;
        if( $square ) {
            $new_width = $new_height = $max_size;
            if( $ratio > 1 ) {
                $src_y = 0;
                $src_x = round( ($width - $height) / 2 );

                $src_w = $src_h = $height;
            } else {
                $src_x = 0;
                $src_y = round( ($height - $width) / 2 );

                $src_w = $src_h = $width;
            }
        } else {
            $src_x = $src_y = 0;
            $src_w = $width;
            $src_h = $height;

            if ( $ratio > 1 ) {
                $new_width  = $max_size;
                $new_height = round( $max_size / $ratio );
            } else {
                $new_height = $max_size;
                $new_width  = round( $max_size * $ratio );
            }
        }
        $func = 'imagecreatefrom' . $type;
        if( !function_exists($func) ) return FALSE;

        $image_src = $func($image_src);
        $new_image = imagecreatetruecolor($new_width,$new_height);

        if( $type=='png' ) {
            imagealphablending($new_image,false);
            if( function_exists('imagesavealpha') )
                imagesavealpha($new_image,true);
        }
        elseif( $type=='gif' && imagecolortransparent($image_src)>=0 ) {
            $transparent_index = imagecolortransparent($image_src);
            $transparent_color = imagecolorsforindex($image_src, $transparent_index);
            $transparent_index = imagecolorallocate($new_image, $transparent_color['red'], $transparent_color['green'], $transparent_color['blue']);
            imagefill($new_image, 0, 0, $transparent_index);
            imagecolortransparent($new_image, $transparent_index);
        }
        imagecopyresampled(
            $new_image, $image_src,
            0, 0, $src_x, $src_y,
            $new_width, $new_height, $src_w, $src_h
        );
        $func = 'image'. $type;
        if($image_dest) {
            if($type == 'png'){
                $func($new_image, $image_dest, 9);
            }else{
                $func($new_image, $image_dest, 90);
            }
        } else {
            header('Content-Type: '. $type_mime);
            $func($new_image);
        }
        imagedestroy($new_image);
        return TRUE;
    }
}


if ( !function_exists("imagecreatefrombmp") ) {

    /**
     * Credit goes to mgutt
     * http://www.programmierer-forum.de/function-imagecreatefrombmp-welche-variante-laeuft-t143137.htm
     * Modified by Fabien Menager to support RGB555 BMP format
     */
    function imagecreatefrombmp($filename) {
        // version 1.1
        if (!($fh = fopen($filename, 'rb'))) {
            trigger_error('imagecreatefrombmp: Can not open ' . $filename, E_USER_WARNING);
            return false;
        }

        // read file header
        $meta = unpack('vtype/Vfilesize/Vreserved/Voffset', fread($fh, 14));

        // check for bitmap
        if ($meta['type'] != 19778) {
            trigger_error('imagecreatefrombmp: ' . $filename . ' is not a bitmap!', E_USER_WARNING);
            return false;
        }

        // read image header
        $meta += unpack('Vheadersize/Vwidth/Vheight/vplanes/vbits/Vcompression/Vimagesize/Vxres/Vyres/Vcolors/Vimportant', fread($fh, 40));
        $bytes_read = 40;

        // read additional bitfield header
        if ($meta['compression'] == 3) {
            $meta += unpack('VrMask/VgMask/VbMask', fread($fh, 12));
            $bytes_read += 12;
        }

        // set bytes and padding
        $meta['bytes'] = $meta['bits'] / 8;
        $meta['decal'] = 4 - (4 * (($meta['width'] * $meta['bytes'] / 4)- floor($meta['width'] * $meta['bytes'] / 4)));
        if ($meta['decal'] == 4) {
            $meta['decal'] = 0;
        }

        // obtain imagesize
        if ($meta['imagesize'] < 1) {
            $meta['imagesize'] = $meta['filesize'] - $meta['offset'];
            // in rare cases filesize is equal to offset so we need to read physical size
            if ($meta['imagesize'] < 1) {
                $meta['imagesize'] = @filesize($filename) - $meta['offset'];
                if ($meta['imagesize'] < 1) {
                    trigger_error('imagecreatefrombmp: Can not obtain filesize of ' . $filename . '!', E_USER_WARNING);
                    return false;
                }
            }
        }

        // calculate colors
        $meta['colors'] = !$meta['colors'] ? pow(2, $meta['bits']) : $meta['colors'];

        // read color palette
        $palette = array();
        if ($meta['bits'] < 16) {
            $palette = unpack('l' . $meta['colors'], fread($fh, $meta['colors'] * 4));
            // in rare cases the color value is signed
            if ($palette[1] < 0) {
                foreach ($palette as $i => $color) {
                    $palette[$i] = $color + 16777216;
                }
            }
        }

        // ignore extra bitmap headers
        if ($meta['headersize'] > $bytes_read) {
            fread($fh, $meta['headersize'] - $bytes_read);
        }

        // create gd image
        $im = imagecreatetruecolor($meta['width'], $meta['height']);
        $data = fread($fh, $meta['imagesize']);

        // uncompress data
        switch ($meta['compression']) {
            case 1: $data = rle8_decode($data, $meta['width']); break;
            case 2: $data = rle4_decode($data, $meta['width']); break;
        }

        $p = 0;
        $vide = chr(0);
        $y = $meta['height'] - 1;
        $error = 'imagecreatefrombmp: ' . $filename . ' has not enough data!';

        // loop through the image data beginning with the lower left corner
        while ($y >= 0) {
            $x = 0;
            while ($x < $meta['width']) {
                switch ($meta['bits']) {
                    case 32:
                    case 24:
                        if (!($part = substr($data, $p, 3 /*$meta['bytes']*/))) {
                            trigger_error($error, E_USER_WARNING);
                            return $im;
                        }
                        $color = unpack('V', $part . $vide);
                        break;
                    case 16:
                        if (!($part = substr($data, $p, 2 /*$meta['bytes']*/))) {
                            trigger_error($error, E_USER_WARNING);
                            return $im;
                        }
                        $color = unpack('v', $part);

                        if (empty($meta['rMask']) || $meta['rMask'] != 0xf800) {
                            $color[1] = (($color[1] & 0x7c00) >> 7) * 65536 + (($color[1] & 0x03e0) >> 2) * 256 + (($color[1] & 0x001f) << 3); // 555
                        }
                        else {
                            $color[1] = (($color[1] & 0xf800) >> 8) * 65536 + (($color[1] & 0x07e0) >> 3) * 256 + (($color[1] & 0x001f) << 3); // 565
                        }
                        break;
                    case 8:
                        $color = unpack('n', $vide . substr($data, $p, 1));
                        $color[1] = $palette[ $color[1] + 1 ];
                        break;
                    case 4:
                        $color = unpack('n', $vide . substr($data, floor($p), 1));
                        $color[1] = ($p * 2) % 2 == 0 ? $color[1] >> 4 : $color[1] & 0x0F;
                        $color[1] = $palette[ $color[1] + 1 ];
                        break;
                    case 1:
                        $color = unpack('n', $vide . substr($data, floor($p), 1));
                        switch (($p * 8) % 8) {
                            case 0: $color[1] =  $color[1] >> 7; break;
                            case 1: $color[1] = ($color[1] & 0x40) >> 6; break;
                            case 2: $color[1] = ($color[1] & 0x20) >> 5; break;
                            case 3: $color[1] = ($color[1] & 0x10) >> 4; break;
                            case 4: $color[1] = ($color[1] & 0x8 ) >> 3; break;
                            case 5: $color[1] = ($color[1] & 0x4 ) >> 2; break;
                            case 6: $color[1] = ($color[1] & 0x2 ) >> 1; break;
                            case 7: $color[1] = ($color[1] & 0x1 );      break;
                        }
                        $color[1] = $palette[ $color[1] + 1 ];
                        break;
                    default:
                        trigger_error('imagecreatefrombmp: ' . $filename . ' has ' . $meta['bits'] . ' bits and this is not supported!', E_USER_WARNING);
                        return false;
                }
                imagesetpixel($im, $x, $y, $color[1]);
                $x++;
                $p += $meta['bytes'];
            }
            $y--;
            $p += $meta['decal'];
        }
        fclose($fh);
        return $im;
    }
}
