<?php
/**
 * Created by IntelliJ IDEA.
 * User: damien
 * Date: 11/02/2017
 * Time: 16:10
 */

namespace CloudFileStorage\Adapter;

use CloudFileStorage\CacheUtil;
use Guzzle\Http\Curl\CurlMultiProxy;
use OpenCloud\Identity\Resource\Token;
use OpenCloud\OpenStack;
use OpenCloud\ObjectStore\Resource;

use CloudFileStorage\AdapterFilesystem;

class OpenCloud implements AdapterFilesystem
{
    private $params;
    private $service;
    private $container;
    private $isDebug=false;
    private $cacheUtil;

    public function __construct($params,$options)
    {
        if($options!= "" && is_array($options) && isset($options["debug"]) && $options["debug"]==true)
            $this->isDebug=true;
        if($options!= "" && is_array($options) && isset($options["cache"]) && isset($options["cacheDir"]) && $options["cache"] == true && $options["cacheDir"] != "")
            $this->cacheUtil=new CacheUtil($options["cacheDir"]);

        $this->params= $params;

        if($this->isDebug)
            $start=microtime(true);

        $client = new OpenStack($params["urlAuth"], array(
            'username' => $params["username"],
            'password'   => $params["password"],
            'tenantName' => $params["tenantName"]
        ),array(\Guzzle\Http\Client::SSL_CERT_AUTHORITY=>false));

        if($this->cacheUtil != null && $this->cacheUtil->cacheIsValid(CacheUtil::CACHE_GETAUTHPROVIDERTOKENS_OPERATION,$params["container"],5*60)){ //5h max d'age du token auth
            $cachedAuth = $this->cacheUtil->getCacheValue(CacheUtil::CACHE_GETAUTHPROVIDERTOKENS_OPERATION,$params["container"]);
            $client->importCredentials(array(
                "token"=>$cachedAuth["token"],
                "expiration"=>$cachedAuth["expiration"],
                "catalog"=>$cachedAuth["catalog"],
            ));
            $client->setTenantObject($cachedAuth["tenant"]);
        }else{
            $client->authenticate();
            /** @var Token $tokenObj */
            $tokenObj = $client->getTokenObject();
            $cacheData = array(
                "catalog"=>$client->getCatalog(),
                "token"=>$tokenObj->getId(),
                "expiration"=>$tokenObj->getExpires(),
                "tenant"=>$client->getTenantObject()
            );
            if($this->cacheUtil != null)
                $this->cacheUtil->saveCacheValue(CacheUtil::CACHE_GETAUTHPROVIDERTOKENS_OPERATION,$params["container"],$cacheData);
        }

        /** @var \OpenCloud\ObjectStore\Service service */
        $this->service = $client->objectStoreService("swift", $params["region"]);

        if($options!= "" && is_array($options) && isset($options["configureSecret"]) && $options["configureSecret"]==true){
            $account = $this->service->getAccount();
            $account->setTempUrlSecret($params["secretTempUrl"]);
        }


        $this->container = $this->service->getContainer($params["container"]);
        if($this->isDebug){
            $end = microtime(true) - $start;
            echo "OpenCloud __construct: ". (round($end,4)*1000)."ms\n<br/>";
        }
    }

    public function listeObjets($path)
    {
        if($this->isDebug)
            $start=microtime(true);

        $listeFichiers = array();

        if($this->cacheUtil != null && $this->cacheUtil->cacheIsValid(CacheUtil::CACHE_LISTDIR_OPTERATION,$path,36*60)) {
            $containerObjectsRaw = $this->cacheUtil->getCacheValue(CacheUtil::CACHE_LISTDIR_OPTERATION,$path);
            if(count($containerObjectsRaw)>0){
                foreach ($containerObjectsRaw as $containerObjectRaw){
                    $containerObject = new \OpenCloud\ObjectStore\Resource\DataObject($this->container,$containerObjectRaw);
                    $listeFichiers[] = new OpenCloudFile($containerObject, $this,array("isDebug"=>$this->isDebug,"cacheUtil"=>$this->cacheUtil));
                }
            }
        }else{
            $toCache=array();
            $containerObjects =  $this->container->objectList(array('prefix' => $path,'delimiter'=>"/"));
            while ($containerObject = $containerObjects->next()) {
                /** @var OpenCloud\ObjectStore\Resource\DataObject $containerObject */
                $ignoreMe = false;
                if($containerObject->getName() == $path || $containerObject->getName()."/" == $path)
                    $ignoreMe = true;

                $tmp = explode("/",$containerObject->getName());
                $tmp = explode(".",end($tmp));
                if(count($tmp) == 1){
                    if($this->endsWith("/",$tmp[0]) == false) {
                        $ignoreMe = false;
                    }else{
                        $ignoreMe = true;
                    }
                }

                if(!$ignoreMe) {//on s'auto liste pas
                    $listeFichiers[] = new OpenCloudFile($containerObject, $this,array("isDebug"=>$this->isDebug,"cacheUtil"=>$this->cacheUtil));
                    $toCache[]=(object)array(
                        "name"=>$containerObject->getName(),
                        "directory"=>$containerObject->isDirectory(),
                        "content_type"=>$containerObject->getContentType(),
                        "bytes"=>$containerObject->getContentLength(),
                        "last_modified"=>$containerObject->getLastModified(),
                        "hash"=>$containerObject->getEtag(),
                    );
                }
            }
            if($this->cacheUtil != null)
                $this->cacheUtil->saveCacheValue(CacheUtil::CACHE_LISTDIR_OPTERATION,$path,$toCache);
        }

        if($this->isDebug){
            $end = microtime(true) - $start;
            echo "OpenCloud listeObjets: ($path) ". (round($end,4)*1000)."ms\n<br/>";
        }

        return $listeFichiers;
    }

    public function getFile($path)
    {
        $raw = $this->getFichierRaw($path);
        return new OpenCloudFile($raw,$this,array("isDebug"=>$this->isDebug,"cacheUtil"=>$this->cacheUtil));
    }
    public function getFichierRaw($path){
        return $this->container->getObject($path);
    }

    public function exists($key)
    {
        return $this->container->objectExists($key);
    }

    public function deleteFile($key)
    {
        try{
            $this->container->deleteObject($key);
            if($this->cacheUtil != null){
                $cacheToDelete = $this->listParentKeyDir($key);
                foreach ($cacheToDelete as $path){
                    if($key.'/' == $path){
                        $path = $key;
                        $this->cacheUtil->deleteCacheValue(CacheUtil::CACHE_INVALIDATEMETADATAFILE_OPERATION, $path);
                    }else{
                        $this->cacheUtil->deleteCacheValue(CacheUtil::CACHE_INVALIDALISTDIR_OPTERATION, $path);

                    }
                }
            }
        }catch (Exception $e){
            //fichier existe pas il semblerait
        }
    }

    public function deleteFiles(Array $arrayKeys)
    {
        $tmp = array();
        foreach ($arrayKeys as $path){
            $tmp[] = $this->container->getName()."/".$path;
        }
        try{
            if($this->cacheUtil != null) {
                foreach ($tmp as $path) {
                    $cacheToDelete = $this->listParentKeyDir($path);
                    foreach ($cacheToDelete as $path) {
                        if($this->endsWith($path, '/')){
                            $path = rtrim($path, '/');
                            $this->cacheUtil->deleteCacheValue(CacheUtil::CACHE_INVALIDALISTDIR_OPTERATION, $path);
                        }else{
                            $this->cacheUtil->deleteCacheValue(CacheUtil::CACHE_INVALIDATEMETADATAFILE_OPERATION, $path);
                        }
                    }
                }

            }
            $this->service->batchDelete($tmp);
        }catch (Exception $e){
            //fichier existe pas il semblerait
        }
    }

    public function copyRemoteFile($key, $destinationKey)
    {
        $fichier = $this->getFichierRaw($key);
        //TODO remove de tous les caches parents de la $destinationKey (exemple /dossier/sousdossier/fichier => delete de cache /dossier/sousdossier/ et /dossier
        if($this->cacheUtil != null){
            $cacheToDelete = $this->listParentKeyDir($destinationKey);
            foreach ($cacheToDelete as $path){
                $this->cacheUtil->deleteCacheValue(CacheUtil::CACHE_INVALIDALISTDIR_OPTERATION, $path);
            }

        }
        $fichier->copy($this->container->getName()."/".$destinationKey);
    }

    public function uploadLocalFile($path, $localPath)
    {
        //TODO gere exceptions si source local existe pas ou si remote existe deja ?
        $containerObject = $this->container->uploadObject($path, fopen($localPath, 'r+'));
        if($this->cacheUtil != null){
            $cacheToDelete = $this->listParentKeyDir($path);
            foreach ($cacheToDelete as $filePath){
                if($path.'/' == $filePath){
                    $filePath = $path;
                    $this->cacheUtil->deleteCacheValue(CacheUtil::CACHE_INVALIDATEMETADATAFILE_OPERATION, $filePath);
                }else{
                    $this->cacheUtil->deleteCacheValue(CacheUtil::CACHE_INVALIDALISTDIR_OPTERATION, $filePath);
                }
            }
        }
        return $this->getFile($path);
    }

    public function generateTemporyUrl($key, \DateTime $dateTimeExpiration, $method, $forceDownload)
    {
        $method = strtoupper($method);
        $expiry = $dateTimeExpiration->getTimestamp();

        // check for proper method
        if ($method != 'GET' && $method != 'PUT') {
            return;
        }

        /** @var Guzzle\Http\Url $urlBase */
        $urlBase = $this->service->getUrl();
        $url = $urlBase->addPath($this->container->getName())->addPath($key);

        $urlPath = urldecode($url->getPath());

        $body = sprintf("%s\n%d\n%s", $method, $expiry, $urlPath);
        $hash = hash_hmac('sha1', $body, $this->params["secretTempUrl"]);

        $extraParam="";
        if(!$forceDownload)
            $extraParam = "&inline";
        return sprintf('%s?temp_url_sig=%s&temp_url_expires=%d%s', $url, $hash, $expiry,$extraParam);
    }

    public static function staticGenerateTemporyUrl($adapterParam, $key, \DateTime $dateTimeExpiration, $method = "GET", $forceDownload=false){
        $method = strtoupper($method);
        $expiry = $dateTimeExpiration->getTimestamp();

        // check for proper method
        if ($method != 'GET' && $method != 'PUT') {
            return;
        }

        $parsedEndpoint = parse_url($adapterParam["urlEndpointSwift"]);
        $urlPath = $parsedEndpoint["path"]."/".$adapterParam["container"]."/".$key;

        $body = sprintf("%s\n%d\n%s", $method, $expiry, $urlPath);
        $hash = hash_hmac('sha1', $body, $adapterParam["secretTempUrl"]);

        $extraParam="";
        if(!$forceDownload)
            $extraParam = "&inline";
        return sprintf('%s?temp_url_sig=%s&temp_url_expires=%d%s', $parsedEndpoint["scheme"]."://".$parsedEndpoint["host"].$urlPath, $hash, $expiry,$extraParam);
    }

    public function createDir( $key)
    {
        if($this->endsWith($key,"/"))
            $key = substr($key, 0, -1);
        $this->container->uploadObject($key, "",array('Content-Type' => 'application/directory'));

        if($this->cacheUtil != null){
            $cacheToDelete = $this->listParentKeyDir($key);
            foreach ($cacheToDelete as $path){
                $this->cacheUtil->deleteCacheValue(CacheUtil::CACHE_INVALIDALISTDIR_OPTERATION, $path);
            }
        }

    }

    public function moveDir($oldPath,$newPath)
    {
        if($this->endsWith($oldPath,"/"))
            $oldPath = substr($oldPath, 0, -1);

        if($this->endsWith($newPath,"/"))
            $newPath = substr($newPath, 0, -1);

        $listeFichiersToMove = $this->listeObjets($oldPath."/");
        $pathToDelete = array();
        foreach ($listeFichiersToMove as $fichier){
            $this->copyRemoteFile($fichier->getPath(),$newPath."/".$fichier->getNom());
            $pathToDelete[]=$fichier->getPath();
        }
        if(!$this->exists($newPath)){
            $this->createDir($newPath);
        }
        $pathToDelete[]=$oldPath;
        $this->deleteFiles($pathToDelete);
    }

    public function deleteDir($key)
    {
        if($this->endsWith($key,"/"))
            $key = substr($key, 0, -1);

        $pathToDelete = array();
        $listeFichiersDossier = $this->listeObjets($key."/");
        foreach ($listeFichiersDossier as $fichier){
            $pathToDelete[]=$fichier->getKey();

        }
        $pathToDelete[]=$key;

        foreach ($pathToDelete as $delete) {
            $cacheToDelete = $this->listParentKeyDir($delete);
            foreach ($cacheToDelete as $path) {
                $this->cacheUtil->deleteCacheValue(CacheUtil::CACHE_INVALIDALISTDIR_OPTERATION, $path);
            }
        }

        $this->deleteFiles($pathToDelete);
    }

    public function endsWith($haystack, $needle) {
        // search forward starting from end minus needle length characters
        return $needle === "" || (($temp = strlen($haystack) - strlen($needle)) >= 0 && strpos($haystack, $needle, $temp) !== false);
    }

    private function listParentKeyDir($key){
        if(!$this->endsWith($key,"/"))
            $key.="/";

        $tmpParentKey = explode("/",$key);
        array_pop($tmpParentKey); //suoppression du dernier element pour remonté d'un niveau
        array_pop($tmpParentKey); //suoppression du dernier element pour remonté d'un niveau
        $folderParent = implode("/",$tmpParentKey);
        $pathToDelete = array($folderParent.'/',$key);
        if($this->isDebug) {
            print_r("listParentKey ($key) ".print_r($pathToDelete,true));
        }
        return $pathToDelete;
    }
}