<?php

namespace CloudFileStorage\Adapter;

use CloudFileStorage\CacheUtil;

use CloudFileStorage\AdapterFilesystem;
use OpenCloud\Common\Constants\Datetime;
use S3;

class S3Cloud implements AdapterFilesystem
{
    private $params;
    /** @var S3 $client */
    public $client;
    public $bucket;
    private $isDebug=false;
    private $cacheUtil;

    public function __construct($params,$options)
    {
        if($options!= "" && is_array($options) && isset($options["debug"]) && $options["debug"]==true)
            $this->isDebug=true;
        if($options!= "" && is_array($options) && isset($options["cache"]) && isset($options["cacheDir"]) && $options["cache"] == true && $options["cacheDir"] != "")
            $this->cacheUtil=new CacheUtil($options["cacheDir"]);

        $this->params= $params;

        if($this->isDebug)
            $start=microtime(true);

        $parsendEndpoint = parse_url($this->params["endpoint"]);
        $isHttps= false;
        if($parsendEndpoint["scheme"] == "https")
            $isHttps = true;
        $this->client = new S3($this->params["accesskey"], $this->params["secretkey"],$isHttps,$parsendEndpoint["host"],$this->params["region"]);
        $this->bucket = $this->params["bucket"];
        if($this->isDebug){
            $end = microtime(true) - $start;
            echo "S3Cloud __construct: ". (round($end,4)*1000)."ms\n<br/>";
        }
    }

    public function listeObjets($path)
    {
        if($this->isDebug)
            $start=microtime(true);

        $listeFichiers = array();

        if($this->cacheUtil != null && $this->cacheUtil->cacheIsValid(CacheUtil::CACHE_LISTDIR_OPTERATION,$path,36*60)) {
            $containerObjectsRaw = $this->cacheUtil->getCacheValue(CacheUtil::CACHE_LISTDIR_OPTERATION,$path);
            if(count($containerObjectsRaw)>0){
                foreach ($containerObjectsRaw as $containerObjectRaw){
                    $listeFichiers[] = new S3CloudFile($containerObjectRaw, $this,array("isDebug"=>$this->isDebug,"cacheUtil"=>$this->cacheUtil));
                }
            }
        }else{
            $toCache=array();
            $containerObjects = $this->client->getBucket($this->bucket,$path,"","","/",true);
            foreach ($containerObjects as $containerObject){
                $listeFichiers[] = new S3CloudFile($containerObject, $this,array("isDebug"=>$this->isDebug,"cacheUtil"=>$this->cacheUtil));
                $toCache[]=$containerObject;

            }

            if($this->cacheUtil != null)
                $this->cacheUtil->saveCacheValue(CacheUtil::CACHE_LISTDIR_OPTERATION,$path,$toCache);
        }

        if($this->isDebug){
            $end = microtime(true) - $start;
            echo "S3Cloud listeObjets: ($path) ". (round($end,4)*1000)."ms\n<br/>";
        }

        return $listeFichiers;
    }

    public function getFile($path)
    {
        $raw = $this->getFichierRaw($path);
        return new S3CloudFile($raw, $this,array("isDebug"=>$this->isDebug,"cacheUtil"=>$this->cacheUtil));
    }
    public function getFichierRaw($path){
        return $this->client->getObjectInfo($this->bucket,$path);
    }

    public function exists($key)
    {
        // Gestion si dossier
        $containerObjects = $this->client->getBucket($this->bucket,$key,"","","/",true);
        $info = false;
        if(isset($containerObjects[$key."/"]) && isset($containerObjects[$key."/"]["prefix"])) {
            $info = true;
        } else {
            $info = $this->client->getObjectInfo($this->bucket, $key);
        }

        if($info)
            return true;
        return false;
    }

    public function deleteFile($key)
    {
        try{
            $this->client->deleteObject($this->bucket,$key);
            if($this->cacheUtil != null){
                $cacheToDelete = $this->listParentKeyDir($key);
                foreach ($cacheToDelete as $path){
                    if($key.'/' == $path){
                        $path = $key;
                        $this->cacheUtil->deleteCacheValue(CacheUtil::CACHE_INVALIDATEMETADATAFILE_OPERATION, $path);
                    }else{
                        $this->cacheUtil->deleteCacheValue(CacheUtil::CACHE_INVALIDALISTDIR_OPTERATION, $path);

                    }
                }
            }
        }catch (Exception $e){
            //fichier existe pas il semblerait
        }
    }

    public function deleteFiles(Array $arrayKeys)
    {
        $tmp = array();
        foreach ($arrayKeys as $path){
            //TODO amélioration a faire en ajoutant dans la classe S3 une méthode de batch delete (https://docs.aws.amazon.com/en_pv/AmazonS3/latest/API/multiobjectdeleteapi.html)
            $this->client->deleteObject($this->bucket,$path);
        }
        try{
            if($this->cacheUtil != null) {
                foreach ($tmp as $path) {
                    $cacheToDelete = $this->listParentKeyDir($path);
                    foreach ($cacheToDelete as $path) {
                        if($this->endsWith($path, '/')){
                            //$path = rtrim($path, '/');
                            $this->cacheUtil->deleteCacheValue(CacheUtil::CACHE_INVALIDALISTDIR_OPTERATION, $path);
                        }else{
                            $this->cacheUtil->deleteCacheValue(CacheUtil::CACHE_INVALIDATEMETADATAFILE_OPERATION, $path);
                        }
                    }
                }

            }
        }catch (Exception $e){
            //fichier existe pas il semblerait
        }
    }


    private function detectContentTypeFromKey($key)
    {
        static $exts = array(
            'jpg' => 'image/jpeg', 'jpeg' => 'image/jpeg', 'gif' => 'image/gif',
            'png' => 'image/png', 'ico' => 'image/x-icon', 'pdf' => 'application/pdf',
            'tif' => 'image/tiff', 'tiff' => 'image/tiff', 'svg' => 'image/svg+xml',
            'svgz' => 'image/svg+xml', 'swf' => 'application/x-shockwave-flash',
            'zip' => 'application/zip', 'gz' => 'application/x-gzip',
            'tar' => 'application/x-tar', 'bz' => 'application/x-bzip',
            'bz2' => 'application/x-bzip2',  'rar' => 'application/x-rar-compressed',
            'exe' => 'application/x-msdownload', 'msi' => 'application/x-msdownload',
            'cab' => 'application/vnd.ms-cab-compressed', 'txt' => 'text/plain',
            'asc' => 'text/plain', 'htm' => 'text/html', 'html' => 'text/html',
            'css' => 'text/css', 'js' => 'text/javascript',
            'xml' => 'text/xml', 'xsl' => 'application/xsl+xml',
            'ogg' => 'application/ogg', 'mp3' => 'audio/mpeg', 'wav' => 'audio/x-wav',
            'avi' => 'video/x-msvideo', 'mpg' => 'video/mpeg', 'mpeg' => 'video/mpeg',
            'mov' => 'video/quicktime', 'flv' => 'video/x-flv', 'php' => 'text/x-php'
        );

        $tmp = explode("/",$key);
        $nom = end($tmp);

        $ext = strtolower(pathinfo($nom, PATHINFO_EXTENSION));
        if (isset($exts[$ext])) return $exts[$ext];

        return "";
    }

    public function copyRemoteFile($key, $destinationKey)
    {
        //TODO remove de tous les caches parents de la $destinationKey (exemple /dossier/sousdossier/fichier => delete de cache /dossier/sousdossier/ et /dossier
        if($this->cacheUtil != null){
            $cacheToDelete = $this->listParentKeyDir($destinationKey);
            foreach ($cacheToDelete as $path){
                $this->cacheUtil->deleteCacheValue(CacheUtil::CACHE_INVALIDALISTDIR_OPTERATION, $path);
            }

        }

        $contentype=$this->detectContentTypeFromKey($destinationKey);
        $this->client->copyObject($this->bucket,$key,$this->bucket,$destinationKey,S3::ACL_PRIVATE,array("Content-Type"=>$contentype));
    }

    public function uploadLocalFile($path, $localPath)
    {
        $localRessource = $this->client->inputFile($localPath);
        $this->client->putObject($localRessource,$this->bucket,$path,S3::ACL_PRIVATE);
        //TODO gere exceptions si source local existe pas ou si remote existe deja ?
        if($this->cacheUtil != null){
            $cacheToDelete = $this->listParentKeyDir($path);
            foreach ($cacheToDelete as $filePath){
                if($path.'/' == $filePath){
                    $filePath = $path;
                    $this->cacheUtil->deleteCacheValue(CacheUtil::CACHE_INVALIDATEMETADATAFILE_OPERATION, $filePath);
                }else{
                    $this->cacheUtil->deleteCacheValue(CacheUtil::CACHE_INVALIDALISTDIR_OPTERATION, $filePath);
                }
            }
        }
        return $this->getFile($path);
    }

    public function generateTemporyUrl($key, \DateTime $dateTimeExpiration, $method, $forceDownload)
    {
        $now = new \Datetime();
        $lifetime =  $dateTimeExpiration->getTimestamp() - $now->getTimestamp();
        return $this->client->getAuthenticatedURL($this->bucket,$key,$lifetime,false,true);
    }

    public static function staticGenerateTemporyUrl($adapterParam, $key, \DateTime $dateTimeExpiration, $method = "GET", $forceDownload=false){
        $parsendEndpoint = parse_url($adapterParam["endpoint"]);
        $isHttps= false;
        if($parsendEndpoint["scheme"] == "https")
            $isHttps = true;
        $client = new S3($adapterParam["accesskey"], $adapterParam["secretkey"],$isHttps,$parsendEndpoint["host"],$adapterParam["region"]);

        $now = new \Datetime();
        $lifetime =  $dateTimeExpiration->getTimestamp() - $now->getTimestamp();
        return $client->getAuthenticatedURL($adapterParam["bucket"],$key,$lifetime,false,true);
    }

    public function createDir( $key)
    {
        if(!$this->endsWith($key,"/"))
            $key .= "/";
        $this->client->putObject("",$this->bucket,$key,S3::ACL_PRIVATE,array(),array("Content-Type"=>"application/directory"));

        if($this->cacheUtil != null){
            $cacheToDelete = $this->listParentKeyDir($key);
            foreach ($cacheToDelete as $path){
                $this->cacheUtil->deleteCacheValue(CacheUtil::CACHE_INVALIDALISTDIR_OPTERATION, $path);
            }
        }
    }

    public function moveDir($oldPath,$newPath)
    {
        if(!$this->endsWith($oldPath,"/"))
            $oldPath .= "/";

        if(!$this->endsWith($newPath,"/"))
            $newPath .= "/";

        if(!$this->exists($newPath)){
            $this->createDir($newPath);
        }

        $listeFichiersToMove = $this->listeObjets($oldPath);
        $pathToDelete = array();
        foreach ($listeFichiersToMove as $fichier){
            $this->copyRemoteFile($fichier->getPath(),$newPath."/".$fichier->getNom());
            $pathToDelete[]=$fichier->getPath();
        }

        $pathToDelete[]=$oldPath;
        $this->deleteFiles($pathToDelete);
    }

    public function deleteDir($key)
    {
        if(!$this->endsWith($key,"/"))
            $key .= "/";

        $pathToDelete = array();
        $listeFichiersDossier = $this->listeObjets($key);
        foreach ($listeFichiersDossier as $fichier){
            $pathToDelete[]=$fichier->getKey();

        }
        $pathToDelete[]=$key;

        foreach ($pathToDelete as $delete) {
            $cacheToDelete = $this->listParentKeyDir($delete);
            foreach ($cacheToDelete as $path) {
                $this->cacheUtil->deleteCacheValue(CacheUtil::CACHE_INVALIDALISTDIR_OPTERATION, $path);
            }
        }

        $this->deleteFiles($pathToDelete);
    }

    public function endsWith($haystack, $needle) {
        // search forward starting from end minus needle length characters
        return $needle === "" || (($temp = strlen($haystack) - strlen($needle)) >= 0 && strpos($haystack, $needle, $temp) !== false);
    }

    private function listParentKeyDir($key){

        if(!$this->endsWith($key,"/"))
            $key.="/";

        $tmpParentKey = explode("/",$key);
        array_pop($tmpParentKey); //suoppression du dernier element pour remonté d'un niveau
        array_pop($tmpParentKey); //suoppression du dernier element pour remonté d'un niveau
        $folderParent = implode("/",$tmpParentKey);
        $pathToDelete = array($folderParent.'/',$key);
        if($this->isDebug) {
            print_r("listParentKey ($key) ".print_r($pathToDelete,true));
        }
        return $pathToDelete;

    }
}