<?php

namespace Actigraph\CarteBundle\Controller;

use Actigraph\CarteBundle\Service\GtfsService;
use Actigraph\CarteBundle\Service\InfoTrafficXmlService;
use App\Entity\Gtfs\Ligne;
use App\Entity\Gtfs\Stop;
use App\Entity\Poi;
use App\Entity\PoiType;
use App\Entity\Token;
use Gelf\Message;
use Gelf\Publisher;
use Gelf\Transport\UdpTransport;
use Psr\Log\LogLevel;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Cache\Adapter\FilesystemAdapter;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Contracts\Cache\CacheInterface;
use Symfony\Contracts\Cache\ItemInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;

/**
 * @Route("/gtfs")
 */
class GtfsController extends AbstractController
{
    /**
     * @Route("/lignes", name="gtfs_lignes", options={"expose"=true})
     */
    public function getLignes(GtfsService $gtfsService)
    {
        $lignes = $gtfsService->getLignes();

        return new JsonResponse($lignes);
    }

    /**
     * @Route("/lignes-arrets", name="gtfs_lignes_arrets", options={"expose"=true})
     */
    public function getLignesWithArrets(GtfsService $gtfsService)
    {
        $lignes = $gtfsService->getLignesWithArrets();

        return new JsonResponse($lignes);
    }

    /**
     * @Route("/ligne/{id}/ordered-arrets/{directionId}", name="gtfs_ligne", options={"expose"=true}, defaults={"directionId"=null})
     */
    public function getLigneWithOrderedArrets($id, $directionId=0, GtfsService $gtfsService)
    {
        /** @var Ligne $ligne */
        $ligne = $gtfsService->getLigneWithOrderedArrets($id, $directionId);

        // Récupératoion des stops logiques présents
        $logicalIds = [];
        $sequenceStops = [];
        foreach ($ligne->getLigneSequences() as $ls) {
            if (!$ls->getIsEnable()) { continue; }
            $stopEntity = $ls->getStopLigne()->getStop();
            $logicalId = $stopEntity->getLogicalStop();
            $sequenceStops[$logicalId] = $stopEntity;
            $logicalIds[] = $logicalId;
        }
        if (empty($logicalIds)) {
            return new JsonResponse($ligne);
        }

        // Récupérer tous les arrêts physiques liés aux stops logiques de la ligne
        /** @var Stop[] $physicals */
        $physicals = $this->getDoctrine()->getRepository(Stop::class)->getPhysicalsByLogical($logicalIds);
        // Indexer par logicalId
        $physByLogical = [];
        foreach ($physicals as $p) {
            $physByLogical[$p->getLogicalStop()][] = $p;
        }

        $routeNumber = (string)$ligne->getNumero();
        $dirFilter = ($directionId === null ? null : (string)$directionId);

        foreach ($sequenceStops as $logicalId => $displayStop) {
            $earliest = null; // ['delay'=>..., 'time'=>int]
            if (!isset($physByLogical[$logicalId])) { continue; }

            foreach ($physByLogical[$logicalId] as $p) {
                $data = $p->getNextPasses();
                if (!is_array($data) || !$data) { continue; }

                foreach ($data as $routeShortName => $directions) {
                    if ((string)$routeShortName !== $routeNumber && (string)$routeShortName !== (string)$id) {
                        continue;
                    }
                    if (!is_array($directions)) { continue; }

                    foreach ($directions as $dirId => $headsigns) {
                        if ($dirFilter !== null && (string)$dirId !== $dirFilter) { continue; }
                        if (!is_array($headsigns)) { continue; }

                        foreach ($headsigns as $tripHeadsign => $trips) {
                            if (!is_array($trips)) { continue; }
                            foreach ($trips as $trip) {
                                if (!isset($trip['time'])) { continue; }
                                $t = is_numeric($trip['time']) ? (int)$trip['time'] : null;
                                if ($t === null) { continue; }

                                if ($earliest === null || $t < $earliest['time']) {
                                    // Utiliser la représentation déjà formatée dans `delay` si disponible
                                    $earliest = [
                                        'delay' => $trip['delay'] ?? null,
                                        'time'  => $t,
                                    ];
                                }
                            }
                        }
                    }
                }
            }

            if ($earliest !== null) {
                $displayStop->addComputedValue('nextPass', $earliest);
            }
        }
        return new JsonResponse($ligne);
    }


    /**
     * @Route("/ligne/{id}/get-all-directions", name="gtfs_ligne_all_directions", options={"expose"=true})
     */
    public function getLigneWithAllDirections($id, GtfsService $gtfsService)
    {
        $ligne = $gtfsService->getLigneWithAllDirections($id);

        return new JsonResponse($ligne);
    }

    /**
     * @Route("/arrets/groupby", name="gtfs_arrets_groupby", options={"expose"=true})
     */
    public function getAllArretsGroupby(GtfsService $gtfsService)
    {
        $results = $gtfsService->getLogicalStops();

        return new JsonResponse($results);
    }

    /**
     * @Route("/info-traffic/{id}", name="gtfs_info_traffic", options={"expose"=true})
     */
    public function getInfoTraffic($id, InfoTrafficXmlService $infoTraffic)
    {
        return new JsonResponse($infoTraffic->getInfoTraffic($id));
    }

    /**
     * @Route("/stopsarea", name="gtfs_stops_area", options={"expose"=true})
     */
    public function getStopsArea(GtfsService $gtfsService)
    {
        $stopsArea = $gtfsService->getLogicalStops();

        return new JsonResponse($stopsArea);
    }

    /**
     * @Route("/stops", name="gtfs_stops", options={"expose"=true})
     */
    public function getStops(GtfsService $gtfsService)
    {
        return new JsonResponse($gtfsService->getPhysicalStops());
    }

    /**
     * @Route("/get-resource/{index}", name="gtfs_get_resource", options={"expose"=true})
     */
    public function getResource($index, $config, HttpClientInterface $client)
    {
        $config = $config['resources'];
        if (!isset($config[$index])) {
            return new JsonResponse('Not found', 404);
        }
        $config = $config[$index];

        $cache = new FilesystemAdapter();
        $data = $cache->get('get_resource_'.$index, function (ItemInterface $item) use ($config, $client, $index) {
            $item->expiresAfter($config['cacheExpireAfter']);
            try {
                if (isset($config['token']) && $config['token'] != '') {
                    $response = $client->request('GET', $config['url'], ['timeout' => 2, 'headers' =>
                        ['X-EXTERNAL-GEO-JSON-API-KEY' => $config['token']],
                    ]);
                } else {
                    $response = $client->request('GET', $config['url'], ['timeout' => 2]);
                }
                $response->getHeaders();
            } catch (\Exception $e) {

            }
            if ($response->getStatusCode() != 200) {
                $item->expiresAfter(30);

                throw new \Exception('Service not available');
            }

            if ($index == 'velostations') {
                $data = json_decode($response->getContent(), 1);
                foreach ($data as $velo) {
                    $externalId = str_replace('urn:ngsi-ld:station:', '', $velo['id']);
                    $persistedVelo = $this->getDoctrine()->getRepository(Poi::class)->findOneBy(['externalId' => $externalId]);
                    if ($persistedVelo == null) {
                        $veloType = $this->getDoctrine()->getRepository(PoiType::class)->find(1);
                        $persistedVelo = new Poi();
                        $location = $velo['location']['value']['coordinates'];
                        $persistedVelo->setExternalId($externalId)
                            ->setCoordLat($location[1])
                            ->setCoordLng($location[0])
                            ->setName($velo['address']['value']['streetAddress'])
                            ->setType($veloType)
                            ->setCoordEdited(false)
                            ->setCoordAngle(0)
                            ->setCoordWidth(0)
                            ->setAdresse($velo['address']['value']['streetAddress'])
                            ->setDescription('Accès au service par :<br>- Carte TaM-abonné  Vélomagg<br>- Application <a title="Téléchargez l\'appli\' M\'Ticket TaM" href="http://bit.ly/AppMTicketTaM" target="_blank">M\'Ticket TaM</a>')
                            ->addOption('bike-available', $velo['availableBikeNumber']['value'])
                            ->addOption('capacity', $velo['totalSlotNumber']['value'])
                            ->addOption('dock-available', $velo['freeSlotNumber']['value'])
                            ->addOption('updated_at', (new \DateTime($velo['availableBikeNumber']['metadata']['timestamp']['value']))->format('d/m/Y à H:i'));

                        $this->getDoctrine()->getManager()->persist($persistedVelo);
                    } else {
                        if ($persistedVelo->getType()->getId() != 1) {
                            return 'ID already taken :'.$externalId;
                        }

                        $persistedVelo->addOption('bike-available', $velo['availableBikeNumber']['value'])
                            ->addOption('capacity', $velo['totalSlotNumber']['value'])
                            ->addOption('dock-available', $velo['freeSlotNumber']['value'])
                            ->addOption('updated_at', (new \DateTime($velo['availableBikeNumber']['metadata']['timestamp']['value']))->format('d/m/Y à H:i'));
                    }
                    $this->getDoctrine()->getManager()->flush();
                }

                return $this->getDoctrine()->getRepository(Poi::class)->findBy(['type' => 1], ['name' => 'ASC']);
            }

            return json_decode($response->getContent(), 1);
        });
        $return = $config['returnClass'];


        return new $return($data);
    }

    /**
     * @Route("/stop/rt/{id}", name="gtfs_stop_rt")
     */
    public function getStopNextPass($id, Request $request)
    {
        $referer = $request->headers->get('referer');
        if (empty($referer) || $referer === '-') {
            return new JsonResponse('Not Found', 404);
        }
        $tokens = $this->getDoctrine()->getRepository(Token::class)->findAll();
        $headerToken = $request->headers->get('X-Api-Key');
        $isValid = false;
        foreach ($tokens as $token) {
            if ($token->getToken() == $headerToken) {
                $isValid = true;
            }
        }
        if (!$isValid) {
            return new JsonResponse('Not Found', 404);
        }

        // Récupération du stop logique et ses stops physiques
        /** @var Stop|null $logical */
        $logical = $this->getDoctrine()->getRepository(Stop::class)->find($id);
        if (!$logical || !$logical->getStopParam()) {
            return new JsonResponse([]);
        }

        $linkedIds = $logical->getStopParam()->getLinkedPhysicalStops();
        if (!is_array($linkedIds) || empty($linkedIds)) {
            $linkedIds = [$id];
        }

        /** @var Stop[] $physicals */
        $physicals = $this->getDoctrine()->getRepository(Stop::class)->findWithIds($linkedIds);

        $merged = [];
        foreach ($physicals as $p) {
            $data = $p->getNextPasses(); // alimenté par la commande cron
            if (!is_array($data) || empty($data)) {
                continue;
            }
            foreach ($data as $routeShortName => $directions) {
                if (!is_array($directions)) { continue; }
                foreach ($directions as $directionId => $headsigns) {
                    if (!is_array($headsigns)) { continue; }
                    foreach ($headsigns as $tripHeadsign => $trips) {
                        if (!is_array($trips)) { continue; }
                        foreach ($trips as $trip) {
                            // Sécuriser la présence de 'time' numérique
                            if (!isset($trip['time'])) { continue; }
                            $t = is_numeric($trip['time']) ? (int)$trip['time'] : null;
                            if ($t === null) { continue; }
                            $merged[$routeShortName][$directionId][$tripHeadsign][] = $trip;
                        }
                    }
                }
            }
        }

        // Tri par ordre des délais
        foreach ($merged as $routeShortName => $directions) {
            foreach ($directions as $directionId => $headsigns) {
                foreach ($headsigns as $tripHeadsign => $list) {
                    usort($list, static function ($a, $b) {
                        return ((int)$a['time']) <=> ((int)$b['time']);
                    });
                    $merged[$routeShortName][$directionId][$tripHeadsign] = $list;
                }
            }
        }

        if (!empty($merged)) {
            ksort($merged, SORT_NATURAL);
        }

        return new JsonResponse($merged);
    }
}
