import * as L from "leaflet"
import * as $ from "jquery"
import { GeoJSONGridLayer } from "./geojsongridlayer";
import {StyleGestion} from "./stylegestion"
import {EventsMap} from "./eventsmap"
import {MapAdmin} from "./admin/mapadmin"
import {Pois} from "./pois"
import {Panneaux} from "./panneaux"
import {Networkstop} from "./networkstop"

require('leaflet-contextmenu');

class BaseCarto {
    contextmenuItems: any[] = [];
    static configuration: any;
    static map : any;
    static plan : any;
    static lineLayer : GeoJSONGridLayer;
    static STYLEGESTION : StyleGestion;
    static MAPADMIN : MapAdmin;
    static POIS:Pois;
    static RENDERPANEL:any;
    static mouvementDisable: boolean  = false;
    constructor(public config: any) {
        BaseCarto.configuration = config;
    }

    public init() {
        let EVENTMAP = new EventsMap(this);
        let STYLE = new StyleGestion(this);
        let POIS = new Pois(this);
        let NETWORKSTOP = new Networkstop();
        let PANNEAUX = new Panneaux(this);

        let plan = L.tileLayer(BaseCarto.configuration.tileUrl, {
            maxZoom: 19,
            attribution: BaseCarto.configuration.creditsTile
        });
        BaseCarto.plan = plan;
        let contextMenu = [{text:"En partir", callback:BaseCarto.RENDERPANEL.enPartir},{text:"Y aller", callback:BaseCarto.RENDERPANEL.yAller},{text:"StreetView", callback:BaseCarto.RENDERPANEL.streetView}];
        let configMap: L.MapOptions = {
            center: [BaseCarto.configuration.center_lat, BaseCarto.configuration.center_lng],
            zoom: BaseCarto.configuration.center_zoom,
            maxZoom: BaseCarto.configuration.max_zoom,
            minZoom: BaseCarto.configuration.min_zoom,
            maxBounds: L.latLngBounds(L.latLng(BaseCarto.configuration.nordouest_lat, BaseCarto.configuration.nordouest_lng), L.latLng(BaseCarto.configuration.sudest_lat, BaseCarto.configuration.sudest_lng)),
            scrollWheelZoom: true,
            layers: [plan],
            contextmenu: true,
            contextmenuWidth: 140,
            contextmenuItems: contextMenu,
            zoomControl: false,
            zoomAnimation: true,
            tap: true
        };

        BaseCarto.map = L.map('map', configMap);

        if(!BaseCarto.configuration.enableDefaultZoomControl && BaseCarto.configuration.addCustomZoomControl) {
            let zoom = L.control.zoom({'position': BaseCarto.configuration.positionZoomCustom});
            zoom.addTo(BaseCarto.map);
        }
        BaseCarto.STYLEGESTION = STYLE;

        if(!BaseCarto.configuration.isIframe) {
            StyleGestion.showOnlyLine = [];
        } else {
            StyleGestion.showOnlyLine = BaseCarto.configuration.importantLines;
        }

        if(BaseCarto.configuration.poisToDisplay.length > 0) {
            POIS.getPois(BaseCarto.configuration.poisToDisplay);
        }
        NETWORKSTOP.getNetworksStop(BaseCarto.configuration.networkStopToDisplay);
        PANNEAUX.getPanneaux(BaseCarto.configuration.panneauxToDisplay);
        EVENTMAP.addMapEvent();

        if(!BaseCarto.configuration.isIframe && !BaseCarto.configuration.adminMode) {
            if(BaseCarto.configuration.hasMinimap){
                L.Control.MiniMap = L.Control.extend({
                    options: {
                        position: 'bottomright',
                        toggleDisplay: false,
                        zoomLevelOffset: -5,
                        zoomLevelFixed: false,
                        zoomAnimation: false,
                        autoToggleDisplay: false,
                        width: 150,
                        height: 100,
                        aimingRectOptions: {color: "#000000", weight: 1, clickable: false},
                        shadowRectOptions: {color: "#000000", weight: 1, clickable: false, opacity:0, fillOpacity:0}
                    },

                    hideText: 'Masquer la miniature',

                    showText: 'Afficher la miniature',

                    initialize: function (layer, options) {
                        L.Util.setOptions(this, options);
                        this.options.aimingRectOptions.clickable = false;
                        this.options.shadowRectOptions.clickable = false;
                        this._layer = layer;
                    },

                    onAdd: function (map) {
                        this._mainMap = map;
                        this._container = L.DomUtil.create('div', 'leaflet-control-minimap');
                        this._container.style.width = this.options.width + 'px';
                        this._container.style.height = this.options.height + 'px';
                        L.DomEvent.disableClickPropagation(this._container);
                        L.DomEvent.on(this._container, 'mousewheel', L.DomEvent.stopPropagation);
                        this._miniMap = new L.Map(this._container,
                            {
                                attributionControl: false,
                                zoomControl: false,
                                zoomAnimation: this.options.zoomAnimation,
                                autoToggleDisplay: this.options.autoToggleDisplay,
                                touchZoom: !this.options.zoomLevelFixed,
                                scrollWheelZoom: !this.options.zoomLevelFixed,
                                doubleClickZoom: !this.options.zoomLevelFixed,
                                boxZoom: !this.options.zoomLevelFixed,
                                crs: map.options.crs
                            });

                        this._miniMap.addLayer(this._layer);

                        this._mainMapMoving = false;
                        this._miniMapMoving = false;

                        this._userToggledDisplay = false;
                        this._minimized = false;

                        if (this.options.toggleDisplay) {
                            this._addToggleButton();
                        }

                        this._miniMap.whenReady(L.Util.bind(function () {
                            this._aimingRect = L.rectangle(this._mainMap.getBounds(), this.options.aimingRectOptions).addTo(this._miniMap);
                            this._shadowRect = L.rectangle(this._mainMap.getBounds(), this.options.shadowRectOptions).addTo(this._miniMap);
                            this._mainMap.on('moveend', this._onMainMapMoved, this);
                            this._mainMap.on('move', this._onMainMapMoving, this);
                            this._miniMap.on('movestart', this._onMiniMapMoveStarted, this);
                            this._miniMap.on('move', this._onMiniMapMoving, this);
                            this._miniMap.on('moveend', this._onMiniMapMoved, this);
                        }, this));

                        return this._container;
                    },

                    addTo: function (map) {
                        L.Control.prototype.addTo.call(this, map);
                        this._miniMap.setView(this._mainMap.getCenter(), this._decideZoom(true));
                        this._setDisplay(this._decideMinimized());
                        return this;
                    },

                    onRemove: function (map) {
                        this._mainMap.off('moveend', this._onMainMapMoved, this);
                        this._mainMap.off('move', this._onMainMapMoving, this);
                        this._miniMap.off('moveend', this._onMiniMapMoved, this);

                        this._miniMap.removeLayer(this._layer);
                    },

                    _addToggleButton: function () {
                        this._toggleDisplayButton = this.options.toggleDisplay ? this._createButton(
                            '', this.hideText, 'leaflet-control-minimap-toggle-display', this._container, this._toggleDisplayButtonClicked, this) : undefined;
                    },

                    _createButton: function (html, title, className, container, fn, context) {
                        var link = L.DomUtil.create('a', className, container);
                        link.innerHTML = html;
                        link.href = '#';
                        link.title = title;

                        var stop = L.DomEvent.stopPropagation;

                        L.DomEvent
                            .on(link, 'click', stop)
                            .on(link, 'mousedown', stop)
                            .on(link, 'dblclick', stop)
                            .on(link, 'click', L.DomEvent.preventDefault)
                            .on(link, 'click', fn, context);

                        return link;
                    },

                    _toggleDisplayButtonClicked: function () {
                        this._userToggledDisplay = true;
                        if (!this._minimized) {
                            this._minimize();
                            this._toggleDisplayButton.title = this.showText;
                        }
                        else {
                            this._restore();
                            this._toggleDisplayButton.title = this.hideText;
                        }
                    },

                    _setDisplay: function (minimize) {
                        if (minimize != this._minimized) {
                            if (!this._minimized) {
                                this._minimize();
                            }
                            else {
                                this._restore();
                            }
                        }
                    },

                    _minimize: function () {
                        if (this.options.toggleDisplay) {
                            this._container.style.width = '19px';
                            this._container.style.height = '19px';
                            this._toggleDisplayButton.className += ' minimized';
                        }
                        else {
                            this._container.style.display = 'none';
                        }
                        this._minimized = true;
                    },

                    _restore: function () {
                        if (this.options.toggleDisplay) {
                            this._container.style.width = this.options.width + 'px';
                            this._container.style.height = this.options.height + 'px';
                            this._toggleDisplayButton.className = this._toggleDisplayButton.className
                                .replace(/(?:^|\s)minimized(?!\S)/g, '');
                        }
                        else {
                            this._container.style.display = 'block';
                        }
                        this._minimized = false;
                    },

                    _onMainMapMoved: function (e) {
                        if (!this._miniMapMoving) {
                            this._mainMapMoving = true;
                            this._miniMap.setView(this._mainMap.getCenter(), this._decideZoom(true));
                            this._setDisplay(this._decideMinimized());
                        } else {
                            this._miniMapMoving = false;
                        }
                        this._aimingRect.setBounds(this._mainMap.getBounds());
                    },

                    _onMainMapMoving: function (e) {
                        this._aimingRect.setBounds(this._mainMap.getBounds());
                    },

                    _onMiniMapMoveStarted:function (e) {
                        var lastAimingRect = this._aimingRect.getBounds();
                        var sw = this._miniMap.latLngToContainerPoint(lastAimingRect.getSouthWest());
                        var ne = this._miniMap.latLngToContainerPoint(lastAimingRect.getNorthEast());
                        this._lastAimingRectPosition = {sw:sw,ne:ne};
                    },

                    _onMiniMapMoving: function (e) {
                        if (!this._mainMapMoving && this._lastAimingRectPosition) {
                            this._shadowRect.setBounds(new L.LatLngBounds(this._miniMap.containerPointToLatLng(this._lastAimingRectPosition.sw),this._miniMap.containerPointToLatLng(this._lastAimingRectPosition.ne)));
                            this._shadowRect.setStyle({opacity:1,fillOpacity:0.3});
                        }
                    },

                    _onMiniMapMoved: function (e) {
                        if (!this._mainMapMoving) {
                            this._miniMapMoving = true;
                            this._mainMap.setView(this._miniMap.getCenter(), this._decideZoom(false));
                            this._shadowRect.setStyle({opacity:0,fillOpacity:0});
                        } else {
                            this._mainMapMoving = false;
                        }
                    },

                    _decideZoom: function (fromMaintoMini) {
                        if (!this.options.zoomLevelFixed) {
                            if (fromMaintoMini)
                                return this._mainMap.getZoom() + this.options.zoomLevelOffset;
                            else {
                                var currentDiff = this._miniMap.getZoom() - this._mainMap.getZoom();
                                var proposedZoom = this._miniMap.getZoom() - this.options.zoomLevelOffset;
                                var toRet;

                                if (currentDiff > this.options.zoomLevelOffset && this._mainMap.getZoom() < this._miniMap.getMinZoom() - this.options.zoomLevelOffset) {
                                    if (this._miniMap.getZoom() > this._lastMiniMapZoom) {
                                        toRet = this._mainMap.getZoom() + 1;
                                        this._miniMap.setZoom(this._miniMap.getZoom() -1);
                                    } else {
                                        toRet = this._mainMap.getZoom();
                                    }
                                } else {
                                    toRet = proposedZoom;
                                }
                                this._lastMiniMapZoom = this._miniMap.getZoom();
                                return toRet;
                            }
                        } else {
                            if (fromMaintoMini)
                                return this.options.zoomLevelFixed;
                            else
                                return this._mainMap.getZoom();
                        }
                    },

                    _decideMinimized: function () {
                        if (this._userToggledDisplay) {
                            return this._minimized;
                        }

                        if (this.options.autoToggleDisplay) {
                            if (this._mainMap.getBounds().contains(this._miniMap.getBounds())) {
                                return true;
                            }
                            return false;
                        }

                        return this._minimized;
                    }
                });
                L.Map.mergeOptions({
                    miniMapControl: false
                });
                L.Map.addInitHook(function () {
                    if (this.options.miniMapControl) {
                        this.miniMapControl = (new L.Control.MiniMap()).addTo(this);
                    }
                });
                L.control.minimap = function (opts) {
                    return new L.Control.MiniMap(opts);
                };

                let planMiniMap = new L.TileLayer(BaseCarto.configuration.tileMiniMapUrl, {
                    minZoom: 0,
                    maxZoom: 13,
                    maxBounds: L.latLngBounds(L.latLng(BaseCarto.configuration.nordouest_lat, BaseCarto.configuration.nordouest_lng), L.latLng(BaseCarto.configuration.sudest_lat, BaseCarto.configuration.sudest_lng)),
                    attribution: BaseCarto.configuration.creditsMiniMapTile
                });
                new L.Control.MiniMap(planMiniMap, {
                    toggleDisplay: true
                }).addTo(BaseCarto.map);
            }
            L.Control.Locate = L.Control.extend({
                options: {
                    position: 'topright',
                    drawCircle: true,
                    follow: false,  // follow with zoom and pan the user's location
                    stopFollowingOnDrag: false, // if follow is true, stop following when map is dragged (deprecated)
                    // range circle
                    circleStyle: {
                        color: '#136AEC',
                        fillColor: '#136AEC',
                        fillOpacity: 0.15,
                        weight: 2,
                        opacity: 0.5
                    },
                    // inner marker
                    markerStyle: {
                        color: '#136AEC',
                        fillColor: '#2A93EE',
                        fillOpacity: 0.7,
                        weight: 2,
                        opacity: 0.9,
                        radius: 5
                    },
                    circlePadding: [0, 0],
                    metric: true,
                    onLocationError: function(err) {
                        alert(err.message);
                    },
                    onLocationOutsideMapBounds: function(context) {
                        setTimeout(function(){
                            alert(context.options.strings.outsideMapBoundsMsg);
                            context.stopLocate();
                        }, 500);

                    },
                    setView: true, // automatically sets the map view to the user's location
                    strings: {
                        title: "Localisation",
                        popup: "You are within {distance} {unit} from this point",
                        outsideMapBoundsMsg: "Votre localisation est trop éloignée du centre ville et ne peut être prise en compte"
                    },
                    locateOptions: {}
                },

                onAdd: function (map) {
                    var container = L.DomUtil.create('div',
                        'leaflet-control-locate leaflet-bar leaflet-control locate');
                    var self = this;
                    this._layer = new L.LayerGroup();
                    this._layer.addTo(map);
                    this._event = undefined;

                    this._locateOptions = {
                        watch: true  // if you overwrite this, visualization cannot be updated
                    };
                    L.extend(this._locateOptions, this.options.locateOptions);
                    L.extend(this._locateOptions, {
                        setView: false // have to set this to false because we have to
                                       // do setView manually
                    });
                    var tmp = {};
                    L.extend(tmp, this.options.markerStyle, this.options.followMarkerStyle);
                    this.options.followMarkerStyle = tmp;
                    tmp = {};
                    L.extend(tmp, this.options.circleStyle, this.options.followCircleStyle);
                    this.options.followCircleStyle = tmp;

                    var link = L.DomUtil.create('a', 'leaflet-bar-part leaflet-bar-part-single leaflet-control-button', container);
                    L.DomUtil.create('span', 'picto-localise', link);
                    link.href = '#';
                    link.title = this.options.strings.title;
                    L.DomEvent
                        .on(link, 'click', L.DomEvent.stopPropagation)
                        .on(link, 'click', L.DomEvent.preventDefault)
                        .on(link, 'click', function() {
                            if (self._active && (self._event === undefined || map.getBounds().contains(self._event.latlng) || !self.options.setView ||
                                    isOutsideMapBounds())) {
                                stopLocate();
                            } else {
                                locate();
                            }
                        })
                        .on(link, 'dblclick', L.DomEvent.stopPropagation);

                    var locate = function () {
                        if (self.options.setView) {
                            self._locateOnNextLocationFound = true;
                        }
                        if(!self._active) {
                            map.locate(self._locateOptions);
                        }
                        self._active = true;
                        if (self.options.follow) {
                            startFollowing();
                        }
                        if (!self._event) {
                            L.DomUtil.addClass(self._container, "requesting");
                            L.DomUtil.removeClass(self._container, "active");
                            L.DomUtil.removeClass(self._container, "following");
                        } else {
                            visualizeLocation();
                        }
                    };

                    var onLocationFound = function (e) {
                        // no need to do anything if the location has not changed
                        if (self._event &&
                            (self._event.latlng.lat === e.latlng.lat &&
                                self._event.latlng.lng === e.latlng.lng &&
                                self._event.accuracy === e.accuracy)) {
                            return;
                        }

                        if (!self._active) {
                            return;
                        }

                        self._event = e;

                        if (self.options.follow && self._following) {
                            self._locateOnNextLocationFound = true;
                        }

                        visualizeLocation();
                    };

                    var startFollowing = function() {
                        map.fire('startfollowing');
                        self._following = true;
                        if (self.options.stopFollowingOnDrag) {
                            map.on('dragstart', stopFollowing);
                        }
                    };

                    var stopFollowing = function() {
                        map.fire('stopfollowing');
                        self._following = false;
                        if (self.options.stopFollowingOnDrag) {
                            map.off('dragstart', stopFollowing);
                        }
                        visualizeLocation();
                    };

                    var isOutsideMapBounds = function () {
                        if (self._event === undefined)
                            return false;
                        return map.options.maxBounds &&
                            !map.options.maxBounds.contains(self._event.latlng);
                    };

                    var visualizeLocation = function() {
                        if (self._event.accuracy === undefined)
                            self._event.accuracy = 0;

                        var radius = self._event.accuracy;
                        if (self._locateOnNextLocationFound) {
                            if (isOutsideMapBounds()) {
                                self.options.onLocationOutsideMapBounds(self);
                            } else {
                                map.fitBounds(self._event.bounds, { padding: self.options.circlePadding });
                            }
                            self._locateOnNextLocationFound = false;
                        }

                        // circle with the radius of the location's accuracy
                        var style, o;
                        if (self.options.drawCircle) {
                            if (self._following) {
                                style = self.options.followCircleStyle;
                            } else {
                                style = self.options.circleStyle;
                            }

                            if (!self._circle) {
                                self._circle = L.circle(self._event.latlng, radius, style)
                                    .addTo(self._layer);
                            } else {
                                self._circle.setLatLng(self._event.latlng).setRadius(radius);
                                for (o in style) {
                                    self._circle.options[o] = style[o];
                                }
                            }
                        }

                        var distance, unit;
                        if (self.options.metric) {
                            distance = radius.toFixed(0);
                            unit = "meters";
                        } else {
                            distance = (radius * 3.2808399).toFixed(0);
                            unit = "feet";
                        }

                        // small inner marker
                        var mStyle;
                        if (self._following) {
                            mStyle = self.options.followMarkerStyle;
                        } else {
                            mStyle = self.options.markerStyle;
                        }

                        var t = self.options.strings.popup;
                        if (!self._circleMarker) {
                            self._circleMarker = L.circleMarker(self._event.latlng, mStyle)
                                .bindPopup(L.Util.template(t, {distance: distance, unit: unit}))
                                .addTo(self._layer);
                        } else {
                            self._circleMarker.setLatLng(self._event.latlng)
                                .bindPopup(L.Util.template(t, {distance: distance, unit: unit}))
                                ._popup.setLatLng(self._event.latlng);
                            for (o in mStyle) {
                                self._circleMarker.options[o] = mStyle[o];
                            }
                        }

                        if (!self._container)
                            return;
                        if (self._following) {
                            L.DomUtil.removeClass(self._container, "requesting");
                            L.DomUtil.addClass(self._container, "active");
                            L.DomUtil.addClass(self._container, "following");
                        } else {
                            L.DomUtil.removeClass(self._container, "requesting");
                            L.DomUtil.addClass(self._container, "active");
                            L.DomUtil.removeClass(self._container, "following");
                        }
                    };

                    var resetVariables = function() {
                        self._active = false;
                        self._locateOnNextLocationFound = self.options.setView;
                        self._following = false;
                    };

                    resetVariables();

                    var stopLocate = function() {
                        map.stopLocate();
                        map.off('dragstart', stopFollowing);
                        L.DomUtil.removeClass(self._container, "requesting");
                        L.DomUtil.removeClass(self._container, "following");
                        resetVariables();
                        self._layer.clearLayers();
                        self._circleMarker = undefined;
                        self._circle = undefined;
                    };

                    var onLocationError = function (err) {
                        // ignore time out error if the location is watched
                        if (err.code == 3 && this._locateOptions.watch) {
                            return;
                        }

                        stopLocate();
                        self.options.onLocationError(err);
                    };

                    // event hooks
                    map.on('locationfound', onLocationFound, self);
                    map.on('locationerror', onLocationError, self);

                    // make locate functions available to outside world
                    this.locate = locate;
                    this.stopLocate = stopLocate;
                    this.stopFollowing = stopFollowing;

                    return container;
                }
            });
            L.control.locate = function (opts) {
                return new L.Control.Locate(opts);
            };
            L.control.locate({position: BaseCarto.configuration.positionPosition}).addTo(BaseCarto.map);

            if(BaseCarto.configuration.hasSelectPoisToDisplay) {
                if (BaseCarto.configuration.poisToDisplay.length > 0) {
                    L.Control.SelectPoi = L.Control.extend({
                        onAdd: function (map) {
                            let div = L.DomUtil.create('div', 'selectPois mt-4');
                            let content = '<div id="selectPois-text" class="d-flex flex-row-reverse"><button type="button" class="btn btn-light shadow-lg" id="choose-poi"><span class="picto-couches"></span>Point d\'intérêts</button></div>';
                            content += '<div id="selectPois-content" class="m-4"><div><p class="text-center m-0">Point d\'intérêts</p></div><hr>';
                            content += '<form>';
                            Pois.pois.forEach(function (e) {
                                let checked = "";
                                if (e.display)
                                    checked = "checked";
                                content += '<div class="custom-control custom-checkbox"><input type="checkbox" class="custom-control-input selectPoi-item" id="selectPoi' + e.id + '" ' + checked + ' data-id="' + e.id + '"><label class="custom-control-label" for="selectPoi' + e.id + '">' + e.name + '</label></div>'
                            });
                            content += '</form>';

                            content += '</div>';
                            div.innerHTML = content;
                            return div;
                        },
                        onRemove: function (map) {
                        }
                    });
                    L.control.selectPoi = function (opts) {
                        return new L.Control.SelectPoi(opts);
                    };
                    L.control.selectPoi({position: 'topright'}).addTo(BaseCarto.map);
                }
            }
            if(BaseCarto.configuration.hasFullScreen) {
                L.Control.FullScreen = L.Control.extend({
                    options: {
                        title: "Plein écran"
                    },

                    onAdd: function (map) {
                        let container = L.DomUtil.create('div',
                            'leaflet-control-locate leaflet-bar leaflet-control fullscreen');
                        let isFullScreen = false;
                        this._layer = new L.LayerGroup();
                        this._layer.addTo(map);
                        this._event = undefined;

                        let tmp = {};
                        L.extend(tmp, this.options.markerStyle, this.options.followMarkerStyle);
                        this.options.followMarkerStyle = tmp;
                        tmp = {};
                        L.extend(tmp, this.options.circleStyle, this.options.followCircleStyle);
                        this.options.followCircleStyle = tmp;

                        let link = L.DomUtil.create('a', 'leaflet-bar-part leaflet-bar-part-single  leaflet-control-button', container);
                        L.DomUtil.create('span', 'picto-direction', link);
                        link.href = '#';
                        link.title = this.options.title;
                        L.DomEvent
                            .on(link, 'click', L.DomEvent.stopPropagation)
                            .on(link, 'click', L.DomEvent.preventDefault)
                            .on(link, 'click', function () {
                                requestFullScreen()
                            })
                            .on(link, 'dblclick', L.DomEvent.stopPropagation)

                        let requestFullScreen = function () {
                            if (!isFullScreen) {
                                isFullScreen = true;
                                document.documentElement.requestFullscreen();
                            } else {
                                if (document.fullscreenElement) {
                                    isFullScreen = false;
                                    document.exitFullscreen();
                                } else {
                                    isFullScreen = true;
                                    document.documentElement.requestFullscreen()
                                }
                            }
                        };
                        return container;
                    },
                });
                L.control.fullscreen = function (opts) {
                    return new L.Control.FullScreen(opts);
                };
                L.control.fullscreen({position: BaseCarto.configuration.positionFullScreen}).addTo(BaseCarto.map);
            }
            if(BaseCarto.configuration.hasReinit){
                L.Control.Raz = L.Control.extend({
                    options: {
                        title: 'Réinitialiser la carte',
                        forceSeparateButton:false
                    },
                    onAdd: function (map) {
                        let className = 'leaflet-control-zoom-raz leaflet-control-button', container;
                        if (map.zoomControl && !this.options.forceSeparateButton) {
                            container = map.zoomControl._container;
                        } else {
                            container = L.DomUtil.create('div', 'leaflet-bar reinit');
                        }

                        this._createButton(this.options.title, className, container, map);

                        return container;
                    },

                    _createButton: function (title, className, container,  context) {
                        let link = L.DomUtil.create('a', className, container);
                        link.href = '#';
                        link.title = title;
                        L.DomUtil.create('span', 'picto-loader ', link);

                        L.DomEvent
                            .addListener(link, 'click', L.DomEvent.stopPropagation)
                            .addListener(link, 'click', L.DomEvent.preventDefault)
                            .addListener(link, 'click', function () {
                                BaseCarto.map.setView(new L.LatLng(BaseCarto.configuration.center_lat, BaseCarto.configuration.center_lng), BaseCarto.configuration.center_zoom);
                            });

                        return link;
                    }
                });

                L.control.raz = function (opts) {
                    return new L.Control.Raz(opts);
                };
                L.control.raz({position: BaseCarto.configuration.positionReinit}).addTo(BaseCarto.map);
            }
            if(BaseCarto.configuration.hasTooglePanel){
                L.Control.ToogleIti = L.Control.extend({
                    options: {
                        title: 'Réduire le panneau',
                        forceSeparateButton:false
                    },
                    onAdd: function (map) {
                        let className = 'leaflet-control-zoom-raz leaflet-control-button', container;
                        container = L.DomUtil.create('div', 'leaflet-bar tooglepanel');
                        this._createButton(this.options.title, className, container, map);

                        return container;
                    },

                    _createButton: function (title, className, container,  context) {
                        let link = L.DomUtil.create('a', className, container);
                        link.href = '#';
                        link.title = title;
                        L.DomUtil.create('span', 'picto-iti', link);

                        L.DomEvent
                            .addListener(link, 'click', L.DomEvent.stopPropagation)
                            .addListener(link, 'click', L.DomEvent.preventDefault)
                            .addListener(link, 'click', BaseCarto.RENDERPANEL.toogleItiPanel);

                        return link;
                    }
                });
                L.control.toogleIti = function (opts) {
                    return new L.Control.ToogleIti(opts);
                };
                L.control.toogleIti({position: BaseCarto.configuration.positionTooglePanel}).addTo(BaseCarto.map);
            }

            $('.leaflet-control-button').attr("data-toggle", "tooltip");
            $('.leaflet-control-button').each(function(index, e){
                let elementParent = $(e).parent();
                if($(elementParent).parent().hasClass("leaflet-right")){
                    $(e).attr("data-placement", "left");
                } else if($(elementParent).parent().hasClass("leaflet-left")){
                    $(e).attr("data-placement", "right");
                }
            })
        }

        let svgLayer = L.svg();
        svgLayer.addTo(BaseCarto.map);

        this.getLines().then(function(lines) {
            let allLines = [];
            lines.forEach(function(e) {
                let identifiant = BaseCarto.configuration.lineprefixMask.replace("{id}", e[BaseCarto.configuration.keyLineObject]);
                allLines[identifiant] = e;
            });
            BaseCarto.lineLayer = new GeoJSONGridLayer(BaseCarto.configuration.tileLinesUrl, STYLE, allLines, {});
            BaseCarto.map.addLayer(BaseCarto.lineLayer);
        });

        if(BaseCarto.configuration.blocage){
            BaseCarto.map.dragging.disable();
            BaseCarto.map.touchZoom.disable();
            BaseCarto.map.doubleClickZoom.disable();
            BaseCarto.map.scrollWheelZoom.disable();
            BaseCarto.mouvementDisable = true;
        }

        if(BaseCarto.configuration.isIframe){
            this.displayMarker("Pois");
            this.displayMarker("networkStop");
            this.updatePanneaux();
        }
    }

    public initAdmin() {
        let ADMIN = new MapAdmin(this);
        let POIS = new Pois(this);
        BaseCarto.POIS = POIS;
        let NETWORKSTOP = new Networkstop();
        let PANNEAUX = new Panneaux(this);
        
        if (BaseCarto.configuration.rightClick.enabled)
            if (BaseCarto.configuration.hasInitineraire && BaseCarto.configuration.rightClick.itineraire) {
                this.contextmenuItems.push({text: "En partir", callback: ""}, {text: "Y aller", callback: ""});

                if (BaseCarto.configuration.rightClick.streetview)
                    this.contextmenuItems.push({text: 'StreetView', callback: ""});

                if (BaseCarto.configuration.rightClick.position != undefined && BaseCarto.configuration.rightClick.position == true)
                    this.contextmenuItems.push({text: 'Position', callback: ""});

            }
        let plan = L.tileLayer(BaseCarto.configuration.tileUrl, {
            maxZoom: 19,
            attribution: '<a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors'
        });
        let contextMenu = [{text:"Ajouter un POI", callback:ADMIN.addPOI}];
        let configMap: L.MapOptions = {
            center: [BaseCarto.configuration.center_lat, BaseCarto.configuration.center_lng],
            zoom: BaseCarto.configuration.center_zoom,
            maxZoom: BaseCarto.configuration.max_zoom,
            minZoom: BaseCarto.configuration.min_zoom,
            maxBounds: L.latLngBounds(L.latLng(BaseCarto.configuration.nordouest_lat, BaseCarto.configuration.nordouest_lng), L.latLng(BaseCarto.configuration.sudest_lat, BaseCarto.configuration.sudest_lng)),
            scrollWheelZoom: true,
            layers: [plan],
            zoomControl: false,
            zoomAnimation: true,
            contextmenu: true,
            contextmenuWidth: 140,
            contextmenuItems: contextMenu
        };

        BaseCarto.map = L.map('map-admin', configMap);
        if(!BaseCarto.configuration.enableDefaultZoomControl && BaseCarto.configuration.addCustomZoomControl) {
            let zoom = L.control.zoom({'position': BaseCarto.configuration.positionZoomCustom});
            zoom.addTo(BaseCarto.map);
        }

        let EVENT = new EventsMap(this);
        BaseCarto.MAPADMIN = ADMIN;
        let STYLE = new StyleGestion(this);
        BaseCarto.STYLEGESTION = STYLE;

        if(!BaseCarto.configuration.isIframe) {
            StyleGestion.showOnlyLine = [];
        } else {
            StyleGestion.showOnlyLine = BaseCarto.configuration.importantLines;
        }

        POIS.getPois(BaseCarto.configuration.poisToDisplay);
        NETWORKSTOP.getNetworksStop(BaseCarto.configuration.networkStopToDisplay);
        PANNEAUX.getPanneaux(BaseCarto.configuration.panneauxToDisplay);
        EVENT.addMapEvent();
        ADMIN.addMapEventAdmin();

        let svgLayer = L.svg();
        svgLayer.addTo(BaseCarto.map);

        this.getLines().then(function(lines) {
            let allLines = [];
            lines.forEach(function(e) {
                let identifiant = BaseCarto.configuration.lineprefixMask.replace("{id}", e[BaseCarto.configuration.keyLineObject]);
                allLines[identifiant] = e;
            });
            BaseCarto.lineLayer = new GeoJSONGridLayer(BaseCarto.configuration.tileLinesUrl, STYLE, allLines, {});
            BaseCarto.map.addLayer(BaseCarto.lineLayer);
        });


    }

    async getLines() {
        try {
            let response = await fetch(BaseCarto.configuration.generateApiRoute("referentiel_lignes", {}));
            let json = await response.json();
            return json;
        }
        catch(e) {
            console.log('Error!', e);
        }
    }

    displayMarker(type:string) {
        let mapBounds = BaseCarto.map.getBounds();
        let markerList = Pois.poisList;
        let layerArray = BaseCarto.configuration.layersPoi;
        let zoomLevel = BaseCarto.configuration.layerZoomLevel;
        if(type === "networkStop") {
            if(Networkstop.isItiStopArea)
                markerList = Networkstop.networkStopListIti;
            else
                markerList = Networkstop.networkStopList;
            
            layerArray = BaseCarto.configuration.networkStopLayer;
            zoomLevel = BaseCarto.configuration.networkLayerZoomLevel
        }
        layerArray.forEach(function(element) {
            let layerConfigConcerned = zoomLevel.find(function(e){ return e.idLayer === element.id});
            let poiConcerned = markerList.find(function(e){ return e.id === layerConfigConcerned.idLayer});
            // Suppression des element de chaque Layer qui ne sont pas dans les bounds
            //console.log(poiConcerned);
            if (typeof poiConcerned !== "undefined"){
                poiConcerned.markerList.forEach(function(marker) {
                    if (mapBounds.contains(marker.getLatLng())) {
                        if (!element.layer.hasLayer(marker)) {
                            marker.addTo(element.layer);
                        }
                    } else {
                        element.layer.removeLayer(marker)
                    }
                });
            }
            // Ajout des layers à la carte en fonction des niveaux de zooms prédéfinis
            if(type === "networkStop"){
                if(Networkstop.isItiStopArea === false) {
                    if(BaseCarto.map.getZoom() >= layerConfigConcerned.zoommin && BaseCarto.map.getZoom() <= layerConfigConcerned.zoommax ) {
                        if (!BaseCarto.map.hasLayer(element.layer)) {
                            BaseCarto.map.addLayer(element.layer);
                        }
                    }else{
                        BaseCarto.map.removeLayer(element.layer);
                    }
                } else {
                    if (!BaseCarto.map.hasLayer(element.layer))
                        BaseCarto.map.addLayer(element.layer);
                }
            } else {
                let poisListConcerned = Pois.pois.find(function(e){ return e.id === layerConfigConcerned.idLayer })
                if(poisListConcerned.display){
                    if(BaseCarto.map.getZoom() >= layerConfigConcerned.zoommin && BaseCarto.map.getZoom() <= layerConfigConcerned.zoommax ) {
                        if (!BaseCarto.map.hasLayer(element.layer)) {
                            BaseCarto.map.addLayer(element.layer);
                        }
                    }else{
                        BaseCarto.map.removeLayer(element.layer);
                    }
                } else {
                    BaseCarto.map.removeLayer(element.layer);
                }
            }
        });
    }

    layerAddRemove(layer){
        if (!BaseCarto.map.hasLayer(layer))
            BaseCarto.map.addLayer(layer);
        else    
            BaseCarto.map.removeLayer(layer);

    }

    updatePanneaux() {
        let ratioPaneaux = 0;
        if (BaseCarto.map.getZoom() == 15) {
            ratioPaneaux = 0.3;
        }
        else if (BaseCarto.map.getZoom() == 16) {
            ratioPaneaux = 0.4;
        }
        else if (BaseCarto.map.getZoom() == 17) {
            ratioPaneaux = 0.5;
        }
        else if (BaseCarto.map.getZoom() >= 18) {
            ratioPaneaux = 1;
        }
        let mapBounds = BaseCarto.map.getBounds();
        let  layerArray = BaseCarto.configuration.panneauxLayer;
        let zoomLevel = BaseCarto.configuration.panneauxLayerZoomLevel;
        let panneauxList = Panneaux.panneauxList;

        layerArray.forEach(function(element){
            let layerConfigConcerned = zoomLevel.find(function(e) { return e.idLayer === element.id });
            let panneauxConcerned = panneauxList.find(function(e) { return  e.id === layerConfigConcerned.idLayer });
            if (typeof panneauxConcerned !== "undefined") {
                panneauxConcerned.markerList.forEach(function(marker) {
                    if (mapBounds.contains(marker.getLatLng())) {
                        if (!element.layer.hasLayer(marker)) {
                            marker.addTo(element.layer);
                        }

                        let newWith = marker.options.width * ratioPaneaux;
                        let newheight = marker.options.height * ratioPaneaux;

                        let markerIcon = L.icon({
                            iconUrl: marker.options.imageurl,
                            iconAnchor: [0, 0],
                            iconSize: [newWith, newheight]
                        });

                        marker.setIcon(markerIcon);
                        marker.setZIndexOffset(-100);

                    } else {
                        element.layer.removeLayer(marker)
                    }

                    if (BaseCarto.map.getZoom() >= layerConfigConcerned.zoommin && BaseCarto.map.getZoom() <= layerConfigConcerned.zoommax) {
                        if (!BaseCarto.map.hasLayer(element.layer))
                            BaseCarto.map.addLayer(element.layer);
                    } else {
                        BaseCarto.map.removeLayer(element.layer);
                    }
                });
            }
        });
    }


}

export { BaseCarto };
