<?php


namespace Actigraph\CarteBundle\Service;


use Actigraph\CarteBundle\Entity\Ligne;
use Actigraph\CarteBundle\Entity\Poi;
use Actigraph\CarteBundle\Entity\PoiType;
use Actigraph\CarteBundle\Entity\StopArea;
use Actigraph\CarteBundle\Entity\Panneau;
use Actigraph\CarteBundle\Entity\Ville;
use Actigraph\CarteBundle\Service\Back\BackAutocomplete;
use Actigraph\CarteBundle\Service\BackObj\RemoteLigne;
use Actigraph\CarteBundle\Service\BackObj\RemoteLigneDirection;
use Actigraph\CarteBundle\Service\BackObj\RemotePointItem;
use Actigraph\CarteBundle\Service\BackObj\RemoteStopArea;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\EntityManagerInterface;
use Psr\Cache\InvalidArgumentException;
use Symfony\Component\Cache\Adapter\FilesystemAdapter;
use Symfony\Component\Cache\Adapter\TagAwareAdapter;
class ReferentielService
{
    private $entityManager;
    public $cache;
    public function __construct(EntityManagerInterface $entityManager)
    {
        $this->entityManager = $entityManager;
        $this->cache = new TagAwareAdapter(new FilesystemAdapter());

    }

    /**
     * @return StopArea[]
     */
    public function listStoparea(){
        $cacheKey = "back.listStopArea";
        $stopAreaList = [];
        try {
            $cacheItem = $this->cache->getItem($cacheKey);
        } catch (InvalidArgumentException $e){
            return $stopAreaList;
        }

        if(!$cacheItem->isHit()){
            $stopAreaList = $this->entityManager->getRepository(StopArea::class)->findAll();
            $cacheItem->set($stopAreaList);
            $cacheItem->expiresAfter(3600);
            $this->cache->save($cacheItem);
        } else {
            $stopAreaList = $cacheItem->get();
        }

        return $stopAreaList;
    }

    /**
     * @return Panneau[]
     */
    public function listPanneaux(){
        $cacheKey = "back.listPanneaux";
        $panneauList = [];
        try {
            $cacheItem = $this->cache->getItem($cacheKey);
        } catch (InvalidArgumentException $e){
            return $panneauList;
        }

        if(!$cacheItem->isHit()){
            $panneauList = $this->entityManager->getRepository(Panneau::class)->findAll();
            $cacheItem->set($panneauList);
            $cacheItem->expiresAfter(3600);
            $this->cache->save($cacheItem);
        } else {
            $panneauList = $cacheItem->get();
        }

        return $panneauList;
    }

    /**
     * @return Ville[]
     */
    public function listVille(){
        $cacheKey = "back.listVille";
        $villeList = [];
        try {
            $cacheItem = $this->cache->getItem($cacheKey);
        } catch (InvalidArgumentException $e){
            return $villeList;
        }

        if(!$cacheItem->isHit()){
            $villeList = $this->entityManager->getRepository(Ville::class)->findAll();
            $cacheItem->set($villeList);
            $cacheItem->expiresAfter(3600);
            $this->cache->save($cacheItem);
        } else {
            $villeList = $cacheItem->get();
        }

        return $villeList;
    }

    /**
     * @return Ligne[]
     */
    public function listLigne(){
        $cacheKey = "back.listLigne";
        $ligneList = [];
        try {
            $cacheItem = $this->cache->getItem($cacheKey);
        } catch (InvalidArgumentException $e){
            return $ligneList;
        }

        if(!$cacheItem->isHit()){
            $ligneList = $this->entityManager->getRepository(Ligne::class)->findBy(array("visible"=>true), array('ordre' => 'ASC'));
            $cacheItem->set($ligneList);
            $cacheItem->expiresAfter(3600);
            $this->cache->save($cacheItem);
        } else {
            $ligneList = $cacheItem->get();
        }

        return $ligneList;
    }

    /**
     * @return PoiType[]
     */
    public function listPoiTypes(){
        $cacheKey = "back.listPoiTypes";
        $poiTypesList = [];
        try {
            $cacheItem = $this->cache->getItem($cacheKey);
        } catch (InvalidArgumentException $e){
            return $poiTypesList;
        }

        if(!$cacheItem->isHit()){
            $poiTypesList = $this->entityManager->getRepository(PoiType::class)->findAll();
            $cacheItem->set($poiTypesList);
            $cacheItem->expiresAfter(3600);
            $this->cache->save($cacheItem);
        } else {
            $poiTypesList = $cacheItem->get();
        }

        return $poiTypesList;
    }

    /**
     * @return PoiType[]
     */
    public function listPoiByTypeId($idType){
        $cacheKey = "back.listPoiByTypeId".$idType;
        $listPoiByTypeId = [];
        try {
            $cacheItem = $this->cache->getItem($cacheKey);
        } catch (InvalidArgumentException $e){
            return $listPoiByTypeId;
        }

        if(!$cacheItem->isHit()){
            $listPoiByTypeId = $this->entityManager->getRepository(Poi::class)->findBy(array("type"=>$idType));
            $cacheItem->set($listPoiByTypeId);
            $cacheItem->expiresAfter(3600);
            $this->cache->save($cacheItem);
        } else {
            $listPoiByTypeId = $cacheItem->get();
        }

        return $listPoiByTypeId;
    }

    /**
     * @return PoiType
     */
    public function getPoi($id){
        return $this->entityManager->getRepository(Poi::class)->findOneBy(array("id"=>$id));
    }

    /**
     * @return StopArea
     */
    public function getStopArea($id){
        return $this->entityManager->getRepository(StopArea::class)->findOneBy(array("id"=>$id));
    }

    /**
     * Recuperation des autocompletions depuis les informations en base
     * @return RemotePointItem[] Liste des suggestions
     */
    public function autocomplete($text): array{
        $suggestions = [];
        /** @var []StopArea $stopAreaMatched */
        $stopAreaMatched = $this->entityManager->getRepository(StopArea::class)->createQueryBuilder('l')
            ->where('l.name LIKE :text')
            ->setParameter('text', "%".$text."%")
            ->getQuery()
            ->getResult();

        if(count($stopAreaMatched)>0){
            /** @var StopArea $stoparea */
            foreach ($stopAreaMatched as $stoparea){
                $city = "";
                if(!is_null($stoparea->getVille()->getName())){
                    $city = " (".$stoparea->getVille()->getName().")";
                }

                $suggestion = new RemotePointItem();
                $suggestion->setType(RemotePointItem::TYPE_STOPAREA);
                $suggestion->setExternalCode($stoparea->getExternalCode());
                $suggestion->setName($stoparea->getName().$city);
                $suggestion->setCoordLat($stoparea->getCoordLat());
                $suggestion->setCoordLng($stoparea->getCoordLng());
                $suggestions[] = $suggestion;
            }
        }

        /** @var []Poi $poiMatched */
        $poiMatched = $this->entityManager->getRepository(Poi::class)->createQueryBuilder('p')
            ->where('p.name LIKE :text')
            ->setParameter('text', "%".$text."%")
            ->getQuery()
            ->getResult();
        if(count($poiMatched)>0){
            /** @var Poi $poi */
            foreach ($poiMatched as $poi){
                $suggestion = new RemotePointItem();
                $suggestion->setType(RemotePointItem::TYPE_POI);
                $suggestion->setTypeExtra($poi->getType()->getId());
                $suggestion->setExternalCode($poi->getExternalCode());
                $suggestion->setName($poi->getName());
                $suggestion->setCoordLat($poi->getCoordLat());
                $suggestion->setCoordLng($poi->getCoordLng());
                $suggestions[] = $suggestion;
            }
        }

        return $suggestions;
    }

    public function mapRemoteLigne(RemoteLigne $remoteLigne){
        $ligneDb = $this->entityManager->getRepository(Ligne::class)->findOneBy(["externalCode"=>$remoteLigne->getExternalCode()]);
        if($ligneDb == null)
            return false;
        return $ligneDb;
    }
    public function mapRemoteLigneWithSomesDirections(RemoteLigne $remoteLigne){
        /** @var Ligne $ligneDb */
        $ligneDbOrigin =  $this->entityManager->getRepository(Ligne::class)->findOneBy(["externalCode"=>$remoteLigne->getExternalCode()]);
        if($ligneDbOrigin == null)
            return "";

        $ligneDb = clone $ligneDbOrigin;
        $dirsForLigne = new ArrayCollection();
        $dirsDb = $ligneDb->getLigneDirections();
        $dirsRemote = $remoteLigne->getDirections();
        $dirsExternalCodeAllowed = [];
        if(count($dirsRemote)>0){
            /** @var RemoteLigneDirection $dirRemote */
            foreach($dirsRemote as $dirRemote){
                $dirsExternalCodeAllowed[]=$dirRemote->getExternalCode();
            }
        }

        if(count($dirsDb)>0){
            foreach ($dirsDb as $dirDb){
                if(in_array($dirDb->getExternalCode(),$dirsExternalCodeAllowed)){
                    $dirsForLigne->add($dirDb);
                }
            }
        }
        $ligneDb->setLigneDirections($dirsDb);
        return $ligneDb;
    }
    public function mapRemoteStoparea(RemoteStopArea $remoteStopArea){
        $stopAreaDb = $this->entityManager->getRepository(StopArea::class)->findOneBy(["externalCode"=>$remoteStopArea->getExternalCode()]);
        if($stopAreaDb == null)
            return false;
        return $stopAreaDb;
    }
}
