<?php


namespace Actigraph\CarteBundle\Service;


use Actigraph\CarteBundle\Entity\Poi;
use Actigraph\CarteBundle\Entity\PoiType;
use Actigraph\CarteBundle\Service\Back\BackAutocomplete;
use Actigraph\CarteBundle\Service\Back\BackInterface;
use Actigraph\CarteBundle\Service\BackObj\Journey;
use Actigraph\CarteBundle\Service\BackObj\RemoteInfoTrafic;
use Actigraph\CarteBundle\Service\BackObj\RemoteLigne;
use Actigraph\CarteBundle\Service\BackObj\RemotePointItem;
use Actigraph\CarteBundle\Service\BackObj\RemoteStopLines;
use Actigraph\CarteBundle\Service\BackObj\RemoteVille;
use Doctrine\ORM\EntityManagerInterface;
use Psr\Log\InvalidArgumentException;

class BackService
{
    public const MODE_TRANSPORTCOMMUN=0;
    public const MODE_VELO=1;
    public const MODE_MARCHEAPIED=2;
    public const MODE_VELOPARTAGE=3;
    public const MODE_VOITURE=4;
    public const MODE_COVOITURAGE=5;

    private $referentielService;
    private $entityManager;
    /** @var [string]BackInterface $backs */
    private $backs;
    private $stopAreaBackClassName="";
    private $poisTypeIdBackClassName=[];
    private $itineraireModeBackClassName=[];
    private $infoTraficsBackClassName="";

    public function __construct(EntityManagerInterface $entityManager,ReferentielService $referentielService,$config)
    {
        $this->entityManager = $entityManager;
        $this->referentielService = $referentielService;
        $listBackOfficeClassNames = [];

        if(isset($config["sync"]["stoparea"])){
            $listBackOfficeClassNames[]=$config["sync"]["stoparea"];
            $this->stopAreaBackClassName=$config["sync"]["stoparea"];
        }

        if(isset($config["infotrafic"])){
            $listBackOfficeClassNames[]=$config["infotrafic"]["class"];
            $this->infoTraficsBackClassName=$config["infotrafic"];
        }

        if(isset($config["sync"]["poi"])) {
            foreach ($config["sync"]["poi"] as $poitypeItem) {
                if (!in_array($poitypeItem["class"], $listBackOfficeClassNames)) {
                    $listBackOfficeClassNames[] = $poitypeItem["class"];
                }
                $extraParams = [];
                if (isset($poitypeItem["params"]) && is_array($poitypeItem["params"]))
                    $extraParams = $poitypeItem["params"];

                $this->poisTypeIdBackClassName[$poitypeItem["id"]] = array(
                    "class" => $poitypeItem["class"],
                    "extraParams" => $extraParams
                );
            }
        }

        foreach ($config["itineraire"] as $itiItem) {
            if(!in_array($itiItem["mode"],$listBackOfficeClassNames)){
                $listBackOfficeClassNames[]=$itiItem["class"];
            }

            $this->itineraireModeBackClassName[$itiItem["mode"]]=array(
                "class"=>$itiItem["class"],
            );

        }

        foreach ($listBackOfficeClassNames as $backName){
            $constructorParam = [];
            if(isset($config["backConfig"][$backName]))
                $constructorParam=$config["backConfig"][$backName];
            $this->backs[$backName]=new $backName($entityManager, $constructorParam);
        }
    }

    /**
     * Recuperation des stoparea depuis le backoffice
     * @return RemoteStopArea[] Liste des arrêts distants
     */
    public function listStoparea() : array {
        return $this->backs[$this->stopAreaBackClassName]->listStoparea();
    }

    /**
     * Recuperation des villes depuis le backoffice
     * @return RemoteVille[] Liste des villes distants
     */
    public function listVilles() : array {
        return $this->backs[$this->stopAreaBackClassName]->listVilles();
    }

    /**
     * Recuperation des lignes depuis le backoffice
     * @return RemoteLigne[] Liste des lignes distants
     */
    public function listLignes(): array{
        return $this->backs[$this->stopAreaBackClassName]->listLignes();
    }

    /**
     * Recuperation des autocomlétions
     * @return RemotePointItem[] Liste des suggestions
     */
    public function autocomplete($text): array{
        $suggestions = [];

        foreach ($this->backs as $back){
            if($back instanceof BackAutocomplete) {
                $suggestions = array_merge($suggestions,$back->autocomplete($text));
            }
        }
        return $suggestions;
    }

    /**
     * Recuperation des infostrafics depuis le backoffice
     * @return RemoteInfoTrafic[] Liste des infotrafics distants
     */
    public function listCurrentInfosTrafics(): array{
        $cacheKey = "back.listCurrentInfosTrafics";
        $perturbationsList = [];
        try {
            $cacheItem = $this->referentielService->cache->getItem($cacheKey);
        } catch (InvalidArgumentException $e){
            return $perturbationsList;
        }

        if(!$cacheItem->isHit()){
            $perturbationsList =  $this->backs[$this->infoTraficsBackClassName["class"]]->listCurrentInfosTrafics();
            //return $perturbations;
            if(count($perturbationsList)>0){
                /** @var RemoteInfoTrafic $perturbation */
                foreach ($perturbationsList as $perturbationIndex=>$perturbation){
                    $perturbLignes = $perturbation->getLignes();
                    if(count($perturbLignes)>0){
                        $mappedLignes = [];
                        foreach ($perturbLignes as $perturbLigne){
                            $mapping= $this->referentielService->mapRemoteLigneWithSomesDirections($perturbLigne);
                            if($mapping){
                                $mappedLignes[] = $mapping;
                            }
                        }
                        $perturbation->setLignes($mappedLignes);
                    }
                    $perturbationsList[$perturbationIndex]=$perturbation;
                }
            }

            $cacheItem->set($perturbationsList);
            $cacheItem->expiresAfter(60);
            $this->referentielService->cache->save($cacheItem);

        } else {
            $perturbationsList = $cacheItem->get();
        }


        return $perturbationsList;
    }

    public function listPoiByTypeId($id){
        /** @var PoiType $type */
        $type = $this->entityManager->getRepository(PoiType::class)->findOneById($id);
        if(!$type)
            return [];
        $poiTypeConfig = $this->poisTypeIdBackClassName[$id];
        return $this->backs[$poiTypeConfig["class"]]->listPoiByTypeExternalCode($type->getExternalCode(),$poiTypeConfig["extraParams"]);
    }

    public function updatePoiExtraData(Poi $poi)
    {

        if(!isset($this->poisTypeIdBackClassName[$poi->getType()->getId()]))
            return false; //Ce type de POI ne supporte pas de données temps réel via un backoffice


        $poiTypeConfig = $this->poisTypeIdBackClassName[$poi->getType()->getId()];

        return $this->backs[$poiTypeConfig["class"]]->updatePoiExtraData($poi,$poi->getType()->getExternalCode(),$poiTypeConfig["extraParams"]);
    }

    public function listUpdatedPoiExtraDataByTypeExternalCode(array $pois,$id)
    {
        /** @var PoiType $type */
        $type = $this->entityManager->getRepository(PoiType::class)->findOneById($id);
        if(!$type)
            return false;
        if(!isset($this->poisTypeIdBackClassName[$id]))
            return false; //Ce type de POI ne supporte pas de données temps réel via un backoffice
        $poiTypeConfig = $this->poisTypeIdBackClassName[$id];
        return $this->backs[$poiTypeConfig["class"]]->listUpdatedPoiExtraDataByTypeExternalCode($pois,$type->getExternalCode(),$poiTypeConfig["extraParams"]);
    }

    public function stopLines($stopId){
        $stopLines =  $this->backs[$this->stopAreaBackClassName]->stopLines($stopId);
        if($stopLines->getLignes() != null && $stopLines->getStopareas() != null){
            $stopArea = $stopLines->getStopareas();
            $mappingStopareas = $this->referentielService->mapRemoteStoparea($stopArea);
            $stopLines->setStopareas($mappingStopareas);
            $allLines = [];
            /** @var RemoteStopLines $stopData */
            foreach ($stopLines->getLignes() as $key=>$stopLine){
                $mappingLines= $this->referentielService->mapRemoteLigneWithSomesDirections($stopLine);
                array_push($allLines, $mappingLines);
            }

            $stopLines->setLignes($allLines);

        }

        return $stopLines;
    }

    public function stopTimetable($stopId, $lineId, $direction, $date){
        $stopTimetable =  $this->backs[$this->stopAreaBackClassName]->getStopTimetable($stopId, $lineId, $direction, $date);
        if($stopTimetable->getLignes() != null && $stopTimetable->getStopareas() != null){
            $stopArea = $stopTimetable->getStopareas();
            $mappingStopareas = $this->referentielService->mapRemoteStoparea($stopArea);
            $stopTimetable->setStopareas($mappingStopareas);
            /** @var RemoteStopLines $stopData */
            $mappingLines= $this->referentielService->mapRemoteLigneWithSomesDirections($stopTimetable->getLignes());
            $stopTimetable->setLignes($mappingLines);
        }
        return $stopTimetable;
    }

    public function lineTimeTable($lineId, $direction){
        $lineTimetable =  $this->backs[$this->stopAreaBackClassName]->getLineTimetable($lineId, $direction);
        if($lineTimetable->getLigne() != null && $lineTimetable->getStopareas() != null){
            $allStopAreas = [];
            /** @var RemoteStopAreas $allStopAreas */
            foreach ($lineTimetable->getStopareas() as $key=>$stopArea){
                $mappingStopArea= $this->referentielService->mapRemoteStoparea($stopArea);
                array_push($allStopAreas, $mappingStopArea);
            }

            /** @var RemoteStopLines $stopData */
            $mappingLine= $this->referentielService->mapRemoteLigneWithSomesDirections($lineTimetable->getLigne());
            $lineTimetable->setLigne($mappingLine);
        }
        return $lineTimetable;
    }

    public function getItineraire($from,$to,$dateTime,$dateTimeRepresent,$modeArray, $extra){
        $allItis = array();
        foreach ($modeArray as $mode) {
            if (!isset($this->itineraireModeBackClassName[$mode]))
                return false;

            if(is_null($mode))
                continue;
            $backConfig = $this->itineraireModeBackClassName[$mode];
            $itis = [];
            if ($mode == self::MODE_TRANSPORTCOMMUN) {
                $itis = $this->backs[$backConfig["class"]]->getJourneyTransportCommun($from, $to, $dateTime, $dateTimeRepresent, $extra, $this->backs[$this->infoTraficsBackClassName["class"]]->listCurrentInfosTrafics($dateTime));
                /** @var Journey $iti */
                foreach ($itis as $iti) {
                    foreach ($iti->getSections() as $section) {
                        if ($section->getTypeJourney() == "TRANSPORT") {
                            $allStopAreas = [];
                            foreach ($section->getStopAreas() as $stopArea) {
                                $mappingStopArea = $this->referentielService->mapRemoteStoparea($stopArea);
                                if(!$mappingStopArea)
                                    $mappingStopArea = $stopArea;
                                array_push($allStopAreas, $mappingStopArea);
                            }
                            $section->setStopAreas($allStopAreas);
                            $mappingLine = $this->referentielService->mapRemoteLigneWithSomesDirections($section->getLignes());
                            $section->setLignes($mappingLine);
                            $perturbations = $section->getPerturbations();
                            if(count($perturbations) > 0){
                                foreach ($perturbations as $perturbationIndex=>$perturbation){
                                    $perturbLignes = $perturbation->getLignes();
                                    if(count($perturbLignes)>0){
                                        $mappedLignes = [];
                                        foreach ($perturbLignes as $perturbLigne){
                                            $mapping= $this->referentielService->mapRemoteLigneWithSomesDirections($perturbLigne);
                                            if($mapping){
                                                $mappedLignes[] = $mapping;
                                            }
                                        }
                                        $perturbation->setLignes($mappedLignes);
                                    }
                                    $perturbations[$perturbationIndex]=$perturbation;
                                }

                                $section->setPerturbations($perturbations);
                            }

                        }
                    }
                }
            } elseif ($mode == self::MODE_VELO) {
                $itis = $this->backs[$backConfig["class"]]->getJourneyVelo($from, $to, $dateTime, $dateTimeRepresent, $extra);
            } elseif ($mode == self::MODE_VOITURE) {
                $itis = $this->backs[$backConfig["class"]]->getJourneyVoiture($from, $to, $dateTime, $dateTimeRepresent, $extra);
            } elseif ($mode == self::MODE_COVOITURAGE) {
                try {
                    $itis = $this->backs[$backConfig["class"]]->getJourneyCovoiturage($from, $to, $dateTime, $dateTimeRepresent, $extra);
                    foreach ($itis as $iti) {
                        foreach ($iti->getSections() as $section) {
                            if ($section->getTypeJourney() == "TRANSPORT") {
                                $allStopAreas = [];
                                $section->setStopAreas($allStopAreas);
                                $mappingLine = $this->referentielService->mapRemoteLigneWithSomesDirections($section->getLignes());
                                $section->setLignes($mappingLine);
                            }
                        }
                    }
                } catch(\Exception $e){
                    $itis = [];
                }
            } elseif ($mode == self::MODE_MARCHEAPIED) {
                $itis = $this->backs[$backConfig['class']]->getJourneyWalk($from, $to, $dateTime, $dateTimeRepresent, $extra);
            }

            foreach ($itis as $iti){
                $allItis[] = $iti;
            }

        }

        return $allItis;
    }

}
