<?php


namespace Actigraph\CarteBundle\Command;

use Actigraph\CarteBundle\Entity\Ligne;
use Actigraph\CarteBundle\Entity\LigneDirection;
use Actigraph\CarteBundle\Entity\LigneType;
use Actigraph\CarteBundle\Entity\Poi;
use Actigraph\CarteBundle\Entity\PoiType;
use Actigraph\CarteBundle\Entity\StopArea;
use Actigraph\CarteBundle\Entity\Ville;
use Actigraph\CarteBundle\Service\BackObj\RemoteLigne;
use Actigraph\CarteBundle\Service\BackObj\RemotePoi;
use Actigraph\CarteBundle\Service\BackObj\RemoteStopArea;
use Actigraph\CarteBundle\Service\BackObj\RemoteVille;
use Actigraph\CarteBundle\Service\BackService;
use Actigraph\CarteBundle\Service\ReferentielService;
use Doctrine\ORM\EntityManagerInterface;
use Psr\Log\InvalidArgumentException;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class SyncBackToReferentiel extends Command
{
    private $entityManager;
    private $backService;
    private $referentielService;
    private $carte_config;

    public function __construct(EntityManagerInterface $entityManager,BackService $backService, ReferentielService $referentielService, $config)
    {
        $this->entityManager = $entityManager;
        $this->backService = $backService;
        $this->carte_config = $config;
        $this->referentielService = $referentielService;
        parent::__construct();
    }

    protected function configure()
    {
        $this
            ->setName('carte:syncbacktoreferentiel')
            ->setDescription('Syncro des backoffice avec le referentiel présent en base de données');
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $correspondanceVilleExternalCodeToId = $this->syncVille();
        $this->syncStopArea($correspondanceVilleExternalCodeToId);
        $this->syncLigne();
        $this->syncPois();

        //$this->backService->listPoiByTypeId(1);
        $output->writeln("Success");
    }


    private function syncVille() :array {
        $correspondance = [];

        /** @var RemoteVille[] $remoteVilles */
        $remoteVilles = $this->backService->listVilles();
        if(count($remoteVilles)==0)
            return $correspondance; //Si j'ai pas de villes alors je fait pas de syncro, surement un probléme dans l'API...
        $repository = $this->entityManager->getRepository(Ville::class);
        $repositoryStopArea = $this->entityManager->getRepository(StopArea::class);

        $villesInBdd=$repository->findAll();
        $villesBddSyncState = [];
        if(is_array($villesInBdd) && count($villesInBdd)>0){
            /** @var Ville $ville */
            foreach ($villesInBdd as $ville){
                $villesBddSyncState[$ville->getId()]=array(
                    "alreadySynced"=>false,
                );
            }
        }

        foreach ($remoteVilles as $remoteVille){
            /** @var Ville $existingVille */
            $existingVille = $repository->findOneBy(array("externalCode"=>$remoteVille->getExternalCode()));
            if(!$existingVille){
                $existingVille = new Ville();
            }
            $existingVille->setCodePostal($remoteVille->getCodePostal());
            $existingVille->setCoordLat($remoteVille->getLat());
            $existingVille->setCoordLng($remoteVille->getLng());
            $existingVille->setExternalCode($remoteVille->getExternalCode());
            $existingVille->setName($remoteVille->getName());

            $this->entityManager->persist($existingVille);
            $this->entityManager->flush();
            $correspondance[$existingVille->getExternalCode()]=$existingVille->getId();

            if(isset($villesBddSyncState[$existingVille->getId()])){
                $villesBddSyncState[$existingVille->getId()]["alreadySynced"]=true;
            }
        }

        if(count($villesBddSyncState)>0){
            foreach ($villesBddSyncState as $villeId=>$villeSyncState){
                if(!$villeSyncState["alreadySynced"]){
                    //Recherche des arrêts associé pour les supprimers également
                    $stopAreaInCity = $repositoryStopArea->findBy(["ville"=>$villeId]);
                    if(count($stopAreaInCity)>0){
                        foreach ( $stopAreaInCity as $stopArea){
                            $this->entityManager->remove($stopArea);
                        }
                        $this->entityManager->flush();
                    }
                    //la ville n'est plus présente dans le remote donc suppression
                    $this->entityManager->remove($this->entityManager->getReference(Ville::class,$villeId));
                }
            }
        }
        $this->entityManager->flush();
        try {
            $this->referentielService->cache->deleteItem("back.listVille");
        } catch (InvalidArgumentException $e){

        }
        return $correspondance;
    }

    private function syncStopArea(array $correspondanceVilleExternalCodeToId)
    {
        /** @var RemoteStopArea[] $remoteStopareas */
        $remoteStopareas = $this->backService->listStoparea();
        if(count($remoteStopareas)==0)
            return; //Si j'ai pas de stoparea alors je fait pas de syncro, surement un probléme dans l'API...
        $repository = $this->entityManager->getRepository(StopArea::class);

        $stopareaInBdd=$repository->findAll();
        $stopareaBddSyncState = [];
        if(is_array($stopareaInBdd) && count($stopareaInBdd)>0){
            /** @var StopArea $stoparea */
            foreach ($stopareaInBdd as $stoparea){
                $stopareaBddSyncState[$stoparea->getId()]=array(
                    "alreadySynced"=>false,
                );
            }
        }

        foreach ($remoteStopareas as $remoteStoparea) {
            /** @var StopArea $existingStoparea */
            $existingStoparea= $repository->findOneBy(array("externalCode"=>$remoteStoparea->getExternalCode()));
            if(!$existingStoparea){
                $existingStoparea = new StopArea();
                $existingStoparea->setCoordEdited(false);
                $existingStoparea->setCoordAngle(0);
                $existingStoparea->setCoordWidth(0);
            }
            $existingStoparea->setName($remoteStoparea->getName());
            $existingStoparea->setExternalCode($remoteStoparea->getExternalCode());
            $existingStoparea->setAccessiblePMR($remoteStoparea->isAccessiblePmr());
            if(!$existingStoparea->getCoordEdited()){
                $existingStoparea->setCoordLng($remoteStoparea->getCoordLng());
                $existingStoparea->setCoordLat($remoteStoparea->getCoordLat());
            }

            if(isset($correspondanceVilleExternalCodeToId[$remoteStoparea->getVilleExternalCode()])){
                $existingStoparea->setVille($this->entityManager->getReference(Ville::class,$correspondanceVilleExternalCodeToId[$remoteStoparea->getVilleExternalCode()]));
            }
            $this->entityManager->persist($existingStoparea);
            $this->entityManager->flush();

            if(isset($stopareaBddSyncState[$existingStoparea->getId()])){
                $stopareaBddSyncState[$existingStoparea->getId()]["alreadySynced"]=true;
            }
        }


        if(count($stopareaBddSyncState)>0){
            foreach ($stopareaBddSyncState as $stopareaId=>$stopareaSyncState){
                if(!$stopareaSyncState["alreadySynced"]){
                    $this->entityManager->remove($this->entityManager->getReference(StopArea::class,$stopareaId));
                }
            }
        }

        try {
            $this->referentielService->cache->deleteItem("back.listStopArea");
        } catch (InvalidArgumentException $e){

        }
        $this->entityManager->flush();
    }

    private function syncLigne() {
        /** @var RemoteVille[] $remoteVilles */
        $remoteLignes = $this->backService->listLignes();
        if(count($remoteLignes)==0)
            return; //Si j'ai pas de lignes alors je fait pas de syncro, surement un probléme dans l'API...
        $repository = $this->entityManager->getRepository(Ligne::class);
        $repositoryDirection = $this->entityManager->getRepository(LigneDirection::class);
        $LignesInBdd=$repository->findAll();
        $lignesBddSyncState = [];
        if(is_array($LignesInBdd) && count($LignesInBdd)>0){
            /** @var Ligne $ligne */
            foreach ($LignesInBdd as $ligne){
                $directionsExternalCode = [];
                $dirsBdd = $ligne->getLigneDirections();
                if(count($dirsBdd)>0){
                    foreach ($dirsBdd as $dirBdd){
                        $directionsExternalCode[]=$dirBdd->getExternalCode();
                    }
                }
                $lignesBddSyncState[$ligne->getId()]=array(
                    "alreadySynced"=>false,
                    "directionsExternalCode"=>$directionsExternalCode
                );
            }
        }
        foreach ($remoteLignes as $remoteLigne){
            /** @var Ligne $existingLigne */
            /** @var RemoteLigne $remoteLigne */
            $existingLigne = $repository->findOneBy(array("externalCode"=>$remoteLigne->getExternalCode()));
            if(!$existingLigne){
                $existingLigne = new Ligne();
                $existingLigne->setVisible(false); //Par defaut la nouvelle ligne n'est pas visible car pas de tracés
                $existingLigne->setOrdre(999);//TODO calcul de l'ordre ?
                $existingLigne->setType($this->entityManager->getReference(LigneType::class,1)); //Premier type par defaut
            }
            $existingLigne->setName($remoteLigne->getName());
            $existingLigne->setShortName($remoteLigne->getShortName());
            $existingLigne->setExternalCode($remoteLigne->getExternalCode());
            $existingLigne->setLigneTextColor($remoteLigne->getLigneTextColor());
            $existingLigne->setLigneBackgroundColor($remoteLigne->getLigneBackgroundColor());
            $params = json_decode($existingLigne->getLigneparams());
            $params->couleur_ligne = '#'.$remoteLigne->getLigneBackgroundColor();
            $params->titre_ligne = ucfirst(strtolower($remoteLigne->getName()));
            $existingLigne->setLigneparams(json_encode($params));
            $existingLigne->setRemoteExist(true);
            //TODO assigner un type ?

            $this->entityManager->persist($existingLigne);
            $this->entityManager->flush();

            //Sync des directions
            $directionsSyncState = [];
            $existingDirections = $existingLigne->getLigneDirections();
            if(count($existingDirections)>0){
                foreach ($existingDirections as $existingDirection){
                    $directionsSyncState[$existingDirection->getId()]=[
                        "alreadySynced"=>false,
                    ];
                }
            }
            if(count($remoteLigne->getDirections())>0){
                foreach ($remoteLigne->getDirections() as $remoteDir){
                    $existingDirection = $repositoryDirection->findOneBy(array("externalCode"=>$remoteDir->getExternalCode()));
                    if(!$existingDirection){
                        $existingDirection = new LigneDirection();
                    }
                    $existingDirection->setName($remoteDir->getName());
                    $existingDirection->setExternalCode($remoteDir->getExternalCode());
                    $existingDirection->setLigne($existingLigne);
                    $this->entityManager->persist($existingDirection);
                    $this->entityManager->flush();

                    if(isset($directionsSyncState[$existingDirection->getId()])){
                        $directionsSyncState[$existingDirection->getId()]["alreadySynced"]=true;
                    }
                }
            }
            foreach ($directionsSyncState as $directionId=>$directionSyncState){
                if(!$directionSyncState["alreadySynced"]){
                    //la ville n'est plus présente dans le remote donc suppression
                    $this->entityManager->remove($this->entityManager->getReference(LigneDirection::class,$directionId));
                }
            }


            if(isset($lignesBddSyncState[$existingLigne->getId()])){
                $lignesBddSyncState[$existingLigne->getId()]["alreadySynced"]=true;
            }
        }
        if(count($lignesBddSyncState)>0){
            foreach ($lignesBddSyncState as $lignesId=>$ligneBddSyncState){
                if(!$ligneBddSyncState["alreadySynced"]){
                    $ligne = $repository->findOneBy(array("id"=>$lignesId));
                    $ligne->setRemoteExist(false);
                    $this->entityManager->persist($ligne);
                }
            }
        }

        try {
            $this->referentielService->cache->deleteItem("back.listLigne");
        } catch (InvalidArgumentException $e){

        }
        $this->entityManager->flush();
    }

    private function syncPois(){
        //Recuperation des rtypes existants en base
        //filtre pour ne garder que ceux ayant une classe de syncro dans le parameter.yaml
        //sync des differents types un par un
        $existingsPoisTypes = $this->entityManager->getRepository(PoiType::class)->findAll();
        $existingsPoisTypesIds=[];
        $poisTypesIdToSync=[];
        /** @var PoiType $existingPoisType */
        foreach ($existingsPoisTypes as $existingPoisType){
            $existingsPoisTypesIds[]=$existingPoisType->getId();
        }
        if(isset($this->carte_config["sync"]["poi"])){
            foreach ($this->carte_config["sync"]["poi"] as $poiConfig){
                if(in_array($poiConfig["id"],$existingsPoisTypesIds)){
                    $poisTypesIdToSync[]=$poiConfig["id"];
                }
            }
            if(count($poisTypesIdToSync)>0){
                foreach ($poisTypesIdToSync as $poiTypeId)
                    $this->syncPoiType($poiTypeId);
            }
        }


        try {
            $this->referentielService->cache->deleteItem("back.listPoiTypes");
        } catch (InvalidArgumentException $e){

        }
    }
    private function syncPoiType($idType){
        $remotePois = $this->backService->listPoiByTypeId($idType);
        if(count($remotePois)==0)
            return; //Si j'ai pas de POis alors je fait pas de syncro, surement un probléme dans l'API...
        $repository = $this->entityManager->getRepository(Poi::class);

        $PoisInBdd=$repository->findBy(array("type"=>$idType));
        $PoisBddSyncState = [];
        if(is_array($PoisInBdd) && count($PoisInBdd)>0){
            /** @var Poi $poi */
            foreach ($PoisInBdd as $poi){
                $PoisBddSyncState[$poi->getId()]=array(
                    "alreadySynced"=>false,
                );
            }
        }

        foreach ($remotePois as $remotePoi) {
            /** @var Poi $existingPoi */
            /** @var RemotePoi $remotePoi */
            $existingPoi = $repository->findOneBy(array("type"=>$idType,"externalCode" => $remotePoi->getExternalCode()));
            if (!$existingPoi) {
                $existingPoi = new Poi();
                $existingPoi->setCoordEdited(false);
                $existingPoi->setCoordAngle(0);
                $existingPoi->setCoordWidth(0);
            }
            $existingPoi->setType($this->entityManager->getReference(PoiType::class,$idType));
            $existingPoi->setName($remotePoi->getName());
            $existingPoi->setExternalCode($remotePoi->getExternalCode());
            $existingPoi->setExtraData($remotePoi->getExtraData());
            if(!$existingPoi->getCoordEdited()){
                $existingPoi->setCoordLng($remotePoi->getLng());
                $existingPoi->setCoordLat($remotePoi->getLat());
            }
            $this->entityManager->persist($existingPoi);
            $this->entityManager->flush();
            if(isset($PoisBddSyncState[$existingPoi->getId()])){
                $PoisBddSyncState[$existingPoi->getId()]["alreadySynced"]=true;
            }
        }

        if(count($PoisBddSyncState)>0){
            foreach ($PoisBddSyncState as $poiId=>$PoiBddSyncState){
                if(!$PoiBddSyncState["alreadySynced"]){
                    $this->entityManager->remove($this->entityManager->getReference(Poi::class,$poiId));
                }
            }
        }
        try {
            $this->referentielService->cache->deleteItem("back.listPoiByTypeId".$idType);
        } catch (InvalidArgumentException $e){

        }

        $this->entityManager->flush();
    }
}
